/*
Rdex version 2.4 for Android

Copyright (C) 2018 Peter Newman <pn@pnewman.com>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE.  See the GNU General Public License for more details:
<http://www.gnu.org/licenses/>.
*/

package com.pnewman.rdex;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.app.DialogFragment;
import android.content.ClipData;
import android.content.ClipboardManager;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.database.Cursor;
import android.graphics.Typeface;
import android.net.Uri;
import android.os.Bundle;
import android.os.Environment;
import android.os.Handler;
import android.os.ParcelFileDescriptor;
import android.provider.OpenableColumns;
import android.support.annotation.NonNull;
import android.text.Editable;
import android.text.Layout;
import android.text.Spannable;
import android.text.method.ScrollingMovementMethod;
import android.text.style.BackgroundColorSpan;
import android.text.style.ForegroundColorSpan;
import android.text.util.Linkify;
import android.view.GestureDetector;
import android.view.GestureDetector.SimpleOnGestureListener;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.View.OnKeyListener;
import android.view.ViewTreeObserver;
import android.view.Window;
import android.view.WindowManager;
import android.view.inputmethod.InputMethodManager;
import android.widget.EditText;
import android.widget.ImageButton;
import android.widget.TextView;
import android.widget.Toast;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;



public class Rdex extends Activity {
    private static final int ACTIVITY_GET_FILENAME = 0;
    private static final int ACTIVITY_DISPLAY_LIST = 1;
    private static final int ACTIVITY_RECENT_FILE_LIST = 2;
    private static final int ACTIVITY_SET_PREFERENCES = 3;
    private static final int ACTIVITY_NEW_CARD = 4;
    private static final int ACTIVITY_EDIT_CARD = 5;
	private static final int ACTIVITY_OPEN_SAF_FILE = 6;
	private static final int ACTIVITY_CREATE_SAF_FILE = 7;
	private static final int ACTIVITY_SAVE_AS_SAF_FILE = 8;
	private static final int DIALOG_NEW_FILE_ID = 0;  // disabled, removed from menu
	private static final int DIALOG_SAVE_AS_ID = 1;
	private static final int DIALOG_DEFAULT_PASSWD_ID = 2;
	private static final int DIALOG_ENCRYPT_PASSWD_ID = 3;
	public static final int DIALOG_DECRYPT_PASSWD_ID = 4;
    public static final String KEY_CARD_ID = "cardId";
    public static final String KEY_CARD_INDEX = "cardIndex";
    public static final String KEY_CARD_TEXT = "cardText";
    public static final String KEY_CARD_TEXT_SIZE = "cardTextSize";
    public static final String KEY_FILE_NAME = "fileName";
	public static final String KEY_CONTENT_URI = "contentUri";
	public static final String KEY_SEARCH_STRING = "searchString";
	public static final String KEY_CRYPTO_SESSION_KEY = "cryptoSessionKey";
	public static final String KEY_CRYPTO_SESSION_KEY_HASH = "cryptoSessionKeyHash";
	public static final String KEY_HMAC_SESSION_KEY = "hmacSessionKey";
	public static final String KEY_HMAC_SESSION_KEY_HASH = "hmacSessionKeyHash";
	public static final String KEY_DISPLAY_LIST = "displayList";
    public static final String KEY_DISPLAY_PAGE_TITLE = "displayPageTitle";
    public static final String KEY_DISPLAY_HELP_FILE = "displayHelpFile";
    public static final String KEY_LIST_SELECTION = "listSelection";
    public static final String KEY_MENU_SELECTION = "menuSelection";
    public static final String KEY_SETTINGS_FONT = "textFont";
    public static final String KEY_SETTINGS_STYLE = "textStyle";
    public static final String KEY_SETTINGS_SIZE = "textSize";
    public static final String KEY_SETTINGS_READ_ONLY_MODE = "readOnlyMode";
    public static final String KEY_SETTINGS_PHONE_MODE = "linkifyPhoneMode";
    public static final String KEY_SETTINGS_WEB_MODE = "linkifyWebMode";
    public static final String KEY_SETTINGS_SHOW_CARD_ID = "showCardIdMode";
    public static final String KEY_SETTINGS_LIST_TITLES_ALPHA = "listTitlesAlphaMode";
	public static final String KEY_SETTINGS_OPEN_IN_LIST_VIEW = "openInListView";
	public static final String KEY_SETTINGS_DEFAULT_PASSWD = "defaultPasswd";
	public static final String KEY_SETTINGS_DEFAULT_PASSWD_HASH = "defaultPasswdHash";
    private static final int HIGHLIGHT_BACKGROUND_COLOR = 0xFF6699DD;
    private static final int HIGHLIGHT_FOREGROUND_COLOR = 0xFF221111;
    private static final int MAX_RECENT_FILES = 5;
    public static final int MAX_WILDCARD_SEARCH = 50;
	public static final int MAX_LIST_ALL_ITEMS = 5000;

    private static final String RDEX_FILE_UTF8_FORMAT_HDR = "\b\007\007 Rdex UTF-8 Format \007\007\b";
	public static final String RDEX_FILE_AES128_FORMAT_HDR = "\b\007 Rdex AES-128-1 File \007\b";
    private static final String RDEX_FILE_UNRECOGNIZED_HDR_01 = "\b\007\007 Rdex";
    public static final String RDEX_FILE_UNRECOGNIZED_HDR_02 = "\b\007 Rdex";

	// UTF-8 byte order mark: 0xEF 0xBB 0xBF
	private static final String UTF8_BYTE_ORDER_MARK = "\357\273\277";

	private enum RdxEncoding {RDX_NONE, RDX_ASCII, RDX_UTF8, RDX_AES128, TXT_ASCII, TXT_UTF8}

	public enum Ternary {TRUE, NEUTRAL, FALSE}

	private EditText mEditText;  // search bar
	private TextView mBodyText;
    private CardList mCardList;
	private RdexCrypto mCrypto;
    private InputMethodManager mInputMgr;
    private String mCurrentFile;
	private Uri mContentUri;
	private String mDisplayName;       // equivalent of filename for content uri
	private RdxEncoding mFileEncoding;
	private ArrayList<Integer> mCardIdList;  //list of card ids from list all cards
	private ArrayList<String> mRecentFileList;
	private boolean mLayoutFinished;
	private boolean mReadOnly;
	private boolean mShowCardId;
	public boolean mListTitlesAlpha;	//display card titles list in alphabetic order
	public boolean mOpenInListView;     //open all files in list all view
	private Handler mHandler;			//one-shot timer
	private Runnable mTimeoutCallback;	//timer callback
	private int mPendingLayout;			//highlight selection when layout complete
	private GestureDetector mGestureDetector;  //used to select text and copy it to clipboard
	private int mCopyIndex;				//offset of last word selected for copy selection
	
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.main);
        
        mBodyText = findViewById(R.id.body);
    	mBodyText.setMovementMethod(new ScrollingMovementMethod());
    	mBodyText.setClickable(false);		//to avoid dimming on scroll
    	mBodyText.setLongClickable(false);	//to avoid dimming on scroll
    	mEditText = findViewById(R.id.searchbar);
    	mInputMgr = (InputMethodManager) getSystemService(Context.INPUT_METHOD_SERVICE);
        mCardList = new CardList(this);
		mCrypto = new RdexCrypto(this, mCardList);
    	mCurrentFile = null;
        mContentUri = null;
		mDisplayName = "";
    	mFileEncoding = RdxEncoding.RDX_NONE;
    	mCardIdList = new ArrayList<>();
    	mRecentFileList = new ArrayList<>();
    	mHandler = new Handler();
    	mPendingLayout = -1;
    	mCopyIndex = -1;

    	mLayoutFinished = false;
	    //prevents showCard(int, int) crashing
	    //because bringPointIntoView() called before layout finished

    	//hides soft keyboard on startup
    	getWindow().setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_STATE_ALWAYS_HIDDEN);

    	//listen for any key press in the edit window
    	mEditText.setOnKeyListener(new OnKeyListener() {
    	    public boolean onKey(View v, int keyCode, KeyEvent event) {
			if (event.getAction() == KeyEvent.ACTION_DOWN) {
				if (keyCode == KeyEvent.KEYCODE_ENTER) {
					listAllCards(mEditText.getText().toString());
					return true;
				}
				return handleMyKeyEvent(keyCode);
			}
			return false;
    	    }
    	});
   	
    	//set listener for the search button
    	final ImageButton button = findViewById(R.id.search_button);
    	button.setOnClickListener(new OnClickListener() {
    	    public void onClick(View v) {
            	mCardList.searchFwd(mEditText.getText().toString());
    	    }
    	});

    	//set listener for the list all button
    	final ImageButton listButton = findViewById(R.id.list_button);
    	listButton.setOnClickListener(new OnClickListener() {
    	    public void onClick(View v) {
    	    	mEditText.setText("");
    	    	listAllCards("");
    	    }
    	});

		//set gesture detection, used to select text and copy it to clipboard
		mGestureDetector = new GestureDetector(this, new MyGestureDetector());
		View.OnTouchListener gestureListener = new View.OnTouchListener() {
			public boolean onTouch(View v, MotionEvent event) {
				// To avoid warning: ontouch should call view#performClick
				v.performClick();
				return mGestureDetector.onTouchEvent(event);
			}
		};
		// (ignore does not override perform click warning)
		mBodyText.setOnTouchListener(gestureListener);

		//restore permanent state
		restorePreferences();

    	//restore previous state, if any
    	//typically occurs if device rotated portrait to landscape or back
    	String fileName = null;
    	String searchString = null;
		Ternary fileLoadSuccess = Ternary.FALSE;
    	int cardIndex = -1;

    	if (savedInstanceState != null) {
			mCrypto.restoreCryptoInstanceState(savedInstanceState);
        	fileName = savedInstanceState.getString(KEY_FILE_NAME);
        	if (fileName != null && fileName.length() > 0) {
				// load file from saved filename
				fileLoadSuccess = loadFile(fileName, true);
			} else {
				// restore uri from saved string and attempt to load content file
				String uriString = savedInstanceState.getString(KEY_CONTENT_URI);
				if (uriString != null && uriString.length() > 0) {
					Uri uri = Uri.parse(uriString);
					fileLoadSuccess = loadContentFromUri(uri, true, false);
				}
			}
			if (fileLoadSuccess == Ternary.TRUE) {
				mCardList.setCurrentCardId(savedInstanceState.getInt(KEY_CARD_ID));
				searchString = savedInstanceState.getString(KEY_SEARCH_STRING);
				if (searchString != null && searchString.length() > 0) {
					mEditText.setText(searchString);
					cardIndex = savedInstanceState.getInt(KEY_CARD_INDEX);
				}
			}
        }
    	else try {
			// check whether incoming intent specifies the file
			boolean contentUri = false;
			Uri uri = getIntent().getData();
			if (uri != null) {
				String scheme = uri.getScheme();
				if (scheme != null && scheme.equals("file")) {
					fileName = uri.getEncodedPath();
				}
				else if (scheme != null && scheme.equals("content")) {
					contentUri = true;
				}
			}

			if (fileName == null && !contentUri) {
				// intent does not specify file so restore previous file
				SharedPreferences settings = getPreferences(0);
				String fileStr = settings.getString("filename0", null);
				if (fileStr != null && fileStr.length() > 0) {
					uri = Uri.parse(fileStr);
					String scheme = null;
					if (uri != null)
						scheme = uri.getScheme();
					if (scheme == null) {
						// string is a filename
						fileName = fileStr;
					}
					else if (scheme.equals("content")) {
						// string is a content uri
						contentUri = true;
					}
				}
			}

			if (fileName != null || contentUri) {
				if (contentUri)
					fileLoadSuccess = loadContentFromUri(uri, false, false);
				else
					fileLoadSuccess = loadFile(fileName, false);
				if (fileLoadSuccess == Ternary.TRUE) {
					openedFileStatusMsg();
				}
			}
    	}
    	catch (ClassCastException e) { /* ignore exception */ }
		catch (Exception e) {
			fileLoadSuccess = Ternary.FALSE;
			mCurrentFile = null;
			mContentUri = null;
			showStatusMsg("Uncaught exception loading file");
		}

    	//callback for GlobalLayoutListener
    	mBodyText.getViewTreeObserver().addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener() {
            @Override
            public void onGlobalLayout() {
                //at this point the layout is complete and the
                //dimensions of mBodyText and any child views are known
            	mLayoutFinished = true;
    			if (mPendingLayout > 0) {
    				mBodyText.bringPointIntoView(mPendingLayout);
    				mPendingLayout = -1;
    			}
            }
        });

    	//callback for timer
    	//bringPointIntoView() fails to work immediately on some systems
    	mTimeoutCallback = new Runnable() {
    		public void run() {
    			if (mLayoutFinished && mPendingLayout > 0) {
    				mBodyText.bringPointIntoView(mPendingLayout);
    			}
    			mPendingLayout = -1;
    		}
    	};

		if (fileLoadSuccess != Ternary.NEUTRAL) {
			// file load operation has completed
			if (mCurrentFile == null && mContentUri == null) {
				showCard(getString(R.string.no_file_loaded));
			} else if (cardIndex > -1) {
				//previous state restored with search string (typically occurs if device rotated)
				mCardList.setCardIndex(cardIndex);
				mCardList.showCard(cardIndex, cardIndex + searchString.length());
			} else {
				//previous state restored, no search string
				displayFileOnOpening();
			}
		}
    }

    class MyGestureDetector extends SimpleOnGestureListener {
        @Override
        public boolean onSingleTapConfirmed(MotionEvent e) {
        	locateText(Math.round(e.getX()), Math.round(e.getY()));
        	return false;
        }
    }

    @Override
    protected void onSaveInstanceState(@NonNull Bundle outState) {
        super.onSaveInstanceState(outState);
        int cardId = mCardList.getCurrentCardId();
        if (cardId < 0) cardId = 0;
        outState.putInt(KEY_CARD_ID, cardId);
        outState.putInt(KEY_CARD_INDEX, mCardList.getCardIndex());
        outState.putString(KEY_FILE_NAME, mCurrentFile);
        outState.putString(KEY_SEARCH_STRING, mEditText.getText().toString());
		String uriString = (mContentUri != null) ? mContentUri.toString() : "";
		outState.putString(KEY_CONTENT_URI, uriString);
		mCrypto.saveCryptoInstanceState(outState);
    }

    //add an exit dialog when back button pressed
	@Override
	public void onBackPressed() {
		if (mFileEncoding != RdxEncoding.RDX_AES128 || mCrypto.isUseDefaultPasswd()) {
			//normal back button behavior
			Rdex.super.onBackPressed();
		} else {
			//exit dialog if file decrypted with passwd entered by user
			AlertDialog.Builder builder = new AlertDialog.Builder(this);

			builder.setTitle("Exit Rdex?");
			builder.setMessage("Are you sure you want to exit Rdex?");
			builder.setPositiveButton("Exit", new DialogInterface.OnClickListener() {
				@Override
				public void onClick(DialogInterface dialog, int id) {
					//normal back button behavior
					Rdex.super.onBackPressed();
				}
			});
			builder.setNegativeButton("Cancel", new DialogInterface.OnClickListener() {
				@Override
				public void onClick(DialogInterface dialog, int id) {
					//ignore back button
				}
			});
			builder.show();
		}
	}

	@Override
	protected void onDestroy() {
		super.onDestroy();
		mCrypto.deleteCurrentPasswd();
	}

	//create main options menu
	@Override
	public boolean onCreateOptionsMenu(Menu menu) {
		MenuInflater inflater = getMenuInflater();
		inflater.inflate(R.menu.rdex_menu, menu);
		return true;
	}


	//respond to menu items selected from the main options menu
	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		if (handleMenuItem(item.getItemId())) return true;
		return super.onOptionsItemSelected(item);
	}


	// (also called by list-all screen menu)
	private boolean handleMenuItem(int itemId) {
    	// case statement removed because resource identifiers are no longer final
		if (itemId == R.id.menu_saf_new_file) {
			createSafFile(ACTIVITY_CREATE_SAF_FILE);
		} else if (itemId == R.id.menu_saf_open_file) {
			openSafFile();
		} else if (itemId == R.id.menu_saf_save_as) {
			saveAsSafFile();
		} else if (itemId == R.id.menu_open_file) {
			openFileDialog();
		} else if (itemId == R.id.menu_save_as) {
			showDialogFragment(DIALOG_SAVE_AS_ID);
		} else if (itemId == R.id.menu_recent_files) {
			openRecentFileDialog();
		} else if (itemId == R.id.menu_new_card) {
			newCard();
		} else if (itemId == R.id.menu_edit_card) {
			editCard();
		} else if (itemId == R.id.menu_delete_card) {
			deleteCard();
		} else if (itemId == R.id.menu_encrypt_file) {
			menuEncryptFile();
		} else if (itemId == R.id.menu_remove_encrypt) {
			menuRemoveFileEncrypt();
		} else if (itemId == R.id.menu_set_passwd) {
			showDialogFragment(DIALOG_DEFAULT_PASSWD_ID);
		} else if (itemId == R.id.menu_preferences) {
			startActivityForResult(new Intent(this, Preferences.class), ACTIVITY_SET_PREFERENCES);
		} else if (itemId == R.id.menu_about) {
			displayAbout();
		} else if (itemId == R.id.menu_help) {
			loadHelpFile();
		} else {
    		return false;
		}

		return true;
	}

	@Override
    public boolean onKeyDown(int keyCode, KeyEvent event) {
        if(event.getAction() == KeyEvent.ACTION_DOWN) {
        	if (handleMyKeyEvent(keyCode)) return true;
        }
        return super.onKeyDown(keyCode, event);
    }

	private boolean handleMyKeyEvent(int keyCode) {
        switch(keyCode) {
        case KeyEvent.KEYCODE_SEARCH:
        	mCardList.searchFwd(mEditText.getText().toString());
            return true;
        case KeyEvent.KEYCODE_DPAD_RIGHT:
        	mCardList.nextCard();
            return true;
        case KeyEvent.KEYCODE_DPAD_LEFT:
        	mCardList.previousCard();
            return true;
        }
        return false;
	}

	private void restorePreferences() {
		SharedPreferences settings = getPreferences(0);
	    int fontSetting = settings.getInt(KEY_SETTINGS_FONT, 0);
	    if (fontSetting < 0 || fontSetting >= Preferences.NUM_FONTS) fontSetting = 0;
	    Typeface tf = Typeface.DEFAULT;
        switch (fontSetting) {
        case 0:
        	tf = Typeface.DEFAULT;
        	break;
        case 1:
        	tf = Typeface.MONOSPACE;
        	break;
        case 2:
        	tf = Typeface.SANS_SERIF;
        	break;
        case 3:
        	tf = Typeface.SERIF;
        	break;
        }
	    int styleSetting = settings.getInt(KEY_SETTINGS_STYLE, 0);
	    if (styleSetting < 0 || styleSetting >= Preferences.NUM_STYLES) styleSetting = 0;
        mBodyText.setTypeface(tf, styleSetting);

	    int sizeSetting = settings.getInt(KEY_SETTINGS_SIZE, Preferences.DEFAULT_TEXT_SIZE);
	    if (sizeSetting < Preferences.SIZE_START_VALUE ||
	    	sizeSetting > Preferences.SIZE_ARRAY_MAX * 2 + Preferences.SIZE_START_VALUE)
	    	sizeSetting = Preferences.DEFAULT_TEXT_SIZE;
        mBodyText.setTextSize((float) sizeSetting);

        mReadOnly =
            settings.getBoolean(KEY_SETTINGS_READ_ONLY_MODE, Preferences.DEFAULT_READ_ONLY_MODE);
        boolean linkifyPhone =
                settings.getBoolean(KEY_SETTINGS_PHONE_MODE, Preferences.DEFAULT_PHONE_MODE);
        boolean linkifyWeb =
                settings.getBoolean(KEY_SETTINGS_WEB_MODE, Preferences.DEFAULT_WEB_MODE);
        int linkifyMask = 0;
        if (linkifyPhone) linkifyMask |= Linkify.PHONE_NUMBERS;
        if (linkifyWeb) linkifyMask |= Linkify.WEB_URLS | Linkify.EMAIL_ADDRESSES;
        mBodyText.setAutoLinkMask(linkifyMask);
        mShowCardId = settings.getBoolean(KEY_SETTINGS_SHOW_CARD_ID, Preferences.DEFAULT_SHOW_CARD_ID);
        mListTitlesAlpha = settings.getBoolean(KEY_SETTINGS_LIST_TITLES_ALPHA, Preferences.DEFAULT_LIST_TITLES_ALPHA);
		mOpenInListView = settings.getBoolean(KEY_SETTINGS_OPEN_IN_LIST_VIEW, Preferences.DEFAULT_OPEN_IN_LIST_VIEW);
        if (!mShowCardId) setScreenTitle();

		String defaultPasswd = settings.getString(KEY_SETTINGS_DEFAULT_PASSWD, null);
		String defaultPasswdHash = settings.getString(KEY_SETTINGS_DEFAULT_PASSWD_HASH, null);
		mCrypto.setDefaultPasswdPref(defaultPasswd, defaultPasswdHash);
	}

	// Dialog used for both filenames and passwords.
	public static class GetStringDialogFragment extends DialogFragment {

		public static GetStringDialogFragment newInstance(int dialogId) {
			GetStringDialogFragment frag = new GetStringDialogFragment();
			Bundle args = new Bundle();
			args.putInt("dialogId", dialogId);
			frag.setArguments(args);
			return frag;
		}

		@Override
		public Dialog onCreateDialog(Bundle savedInstanceState) {
			final int dialogId = getArguments().getInt("dialogId");
			int dialogTitle, dialogMsg, layoutSrc;
			switch (dialogId) {
				case DIALOG_NEW_FILE_ID:
					// disabled, removed from menu
					dialogTitle = R.string.new_file_dialog_title;
					dialogMsg = R.string.new_file_dialog;
					layoutSrc = R.layout.dialog_get_string;
					break;
				case DIALOG_ENCRYPT_PASSWD_ID:
					dialogTitle = R.string.encrypt_passwd_dialog_title;
					dialogMsg = R.string.encrypt_passwd_dialog;
					layoutSrc = R.layout.dialog_get_string;
					break;
				case DIALOG_DECRYPT_PASSWD_ID:
					dialogTitle = R.string.decrypt_passwd_dialog_title;
					dialogMsg = R.string.decrypt_passwd_dialog;
					layoutSrc = R.layout.dialog_get_passwd;
					break;
				case DIALOG_SAVE_AS_ID:
				default:
					dialogTitle = R.string.save_as_dialog_title;
					dialogMsg = R.string.save_as_dialog;
					layoutSrc = R.layout.dialog_get_string;
					break;
			}

			LayoutInflater inflater = getActivity().getLayoutInflater();
			final View layout = inflater.inflate(layoutSrc, null);
			final EditText editTextBox = layout.findViewById(R.id.EditTextBox);
			((TextView) layout.findViewById(R.id.TextViewMsg)).setText(dialogMsg);

			AlertDialog.Builder builder = new AlertDialog.Builder(getActivity());
			builder.setTitle(dialogTitle);
			builder.setView(layout);

			//listen for enter key press in the edit window
			editTextBox.setOnKeyListener(new OnKeyListener() {
				public boolean onKey(View v, int keyCode, KeyEvent event) {
					if (event.getAction() == KeyEvent.ACTION_DOWN) {
						if (keyCode == KeyEvent.KEYCODE_ENTER) {
							((Rdex) getActivity()).dialogResponse(dialogId, true, editTextBox.getText());
							editTextBox.setText("");
							dismiss();
							return true;
						}
					}
					return false;
				}
			});
			builder.setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int which) {
					((Rdex)getActivity()).dialogResponse(dialogId, true, editTextBox.getText());
					editTextBox.setText("");
				}
			});
			builder.setNegativeButton(android.R.string.cancel, new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int whichButton) {
					((Rdex)getActivity()).dialogResponse(dialogId, false, null);
					editTextBox.setText("");
				}
			});

			AlertDialog dialog = builder.create();
			Window window = dialog.getWindow();
			if (window != null) window.setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_STATE_VISIBLE);
			editTextBox.requestFocus();
			return dialog;
		}
	}

	public static class DefaultPasswdDialogFragment extends DialogFragment {

		public static DefaultPasswdDialogFragment newInstance() {
			return new DefaultPasswdDialogFragment();
		}

		@Override
		public Dialog onCreateDialog(Bundle savedInstanceState) {
			LayoutInflater inflater = getActivity().getLayoutInflater();
			final View layout = inflater.inflate(R.layout.dialog_get_string, null);
			final EditText editPasswd = layout.findViewById(R.id.EditTextBox);
			((TextView) layout.findViewById(R.id.TextViewMsg)).setText(R.string.default_passwd_dialog);

			AlertDialog.Builder builder = new AlertDialog.Builder(getActivity());
			builder.setTitle(R.string.default_passwd_dialog_title);
			builder.setView(layout);

			//listen for enter key press in the edit window
			editPasswd.setOnKeyListener(new OnKeyListener() {
				public boolean onKey(View v, int keyCode, KeyEvent event) {
					if (event.getAction() == KeyEvent.ACTION_DOWN) {
						if (keyCode == KeyEvent.KEYCODE_ENTER) {
							((Rdex)getActivity()).mCrypto.setDefaultPasswdResult(editPasswd.getText());
							editPasswd.setText("");
							dismiss();
							return true;
						}
					}
					return false;
				}
			});
			builder.setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int id) {
					((Rdex)getActivity()).mCrypto.setDefaultPasswdResult(editPasswd.getText());
					editPasswd.setText("");
				}
			});
			builder.setNegativeButton(R.string.clear_button, new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int id) {
					((Rdex)getActivity()).mCrypto.setDefaultPasswdResult(null);
					editPasswd.setText("");
				}
			});
			builder.setNeutralButton(android.R.string.cancel, new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int id) {
					editPasswd.setText("");
				}
			});

			AlertDialog dialog = builder.create();
			Window window = dialog.getWindow();
			if (window != null) window.setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_STATE_VISIBLE);
			editPasswd.requestFocus();
			return dialog;
		}
	}

	private void dialogResponse(int dialogId, boolean result, Editable responseText) {
		switch (dialogId) {
			case DIALOG_SAVE_AS_ID:
			case DIALOG_NEW_FILE_ID:
				if (result) checkFilename(responseText.toString(), dialogId);
				break;
			case DIALOG_ENCRYPT_PASSWD_ID:
				if (result && mCrypto.setCryptoPasswdResult(responseText)) {
					mCrypto.resetUseDefaultPasswd();
					mFileEncoding = RdxEncoding.RDX_AES128;
					saveFile();
				}
				break;
			case DIALOG_DECRYPT_PASSWD_ID:
				rdexDecryptDialogResponse(result, responseText);
				break;
		}
	}

	public void showDialogFragment(int dialogId) {
		DialogFragment newDialogFragment;
		if (dialogId == DIALOG_DEFAULT_PASSWD_ID) {
			newDialogFragment = DefaultPasswdDialogFragment.newInstance();
		} else {
			newDialogFragment = GetStringDialogFragment.newInstance(dialogId);
		}
		newDialogFragment.show(getFragmentManager(), "dialogFragment_" + dialogId);
	}

	// encrypt file menu item
	private void menuEncryptFile() {
		if (mCurrentFile == null && mContentUri == null) {
			alertMsg("There is no file currently open. You must first open a file before configuring encryption for it.");
			return;
		}
		new AlertDialog.Builder(this)
		.setTitle("Use Default Passphrase?")
		.setMessage(R.string.encrypt_passwd_default_dialog)
		.setPositiveButton("Yes", new DialogInterface.OnClickListener() {
			@Override
			public void onClick(DialogInterface dialog, int id) {
			if (mCrypto.isDefaultPasswdSet()) {
				mCrypto.setUseDefaultPasswd();
				mFileEncoding = RdxEncoding.RDX_AES128;
				saveFile();
			} else {
				alertMsg("The default passphrase has not been set. You must first set the default passphrase " +
					"(from the Encryption menu) before it can be used to encrypt a file.");
			}
			}
		})
		.setNeutralButton("Cancel", new DialogInterface.OnClickListener() {
			@Override
			public void onClick(DialogInterface dialog, int id) { }
		})
		.setNegativeButton("No", new DialogInterface.OnClickListener() {
			@Override
			public void onClick(DialogInterface dialog, int id) { showDialogFragment(DIALOG_ENCRYPT_PASSWD_ID); }
		})
		.show();
	}

	// remove file encryption menu item
	private void menuRemoveFileEncrypt() {
		if (mCurrentFile == null && mContentUri == null) {
			alertMsg("There is no file currently open. You must first open an encrypted file before disabling encryption on it.");
		}
		else if (mFileEncoding != RdxEncoding.RDX_AES128) {
			alertMsg("The file currently open is not in an encrypted format. " +
					"You can only remove encryption from an encrypted file.");
		} else {
			mFileEncoding = RdxEncoding.RDX_UTF8;
			saveFile();
		}
	}

	private void saveAsSafFile() {
		new AlertDialog.Builder(this)
				.setMessage(R.string.save_as_saf_dialog)
				.setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {
					@Override
					public void onClick(DialogInterface dialog, int which) {
						createSafFile(ACTIVITY_SAVE_AS_SAF_FILE);
					}
				})
				.setNegativeButton(android.R.string.cancel, new DialogInterface.OnClickListener() {
					@Override
					public void onClick(DialogInterface dialog, int which) { }
				})
				.show();
	}

	// response from decrypt password dialog
	// Android dialogs are not modal so finish decryption of the file and update state
	public void rdexDecryptDialogResponse(boolean response, Editable passwd) {
		Ternary result = mCrypto.decryptDialogResponse(response, true, passwd);
		if (result == Rdex.Ternary.TRUE) {
			mFileEncoding = RdxEncoding.RDX_AES128;
			displayFileOnOpening();
			openedFileStatusMsg();
		}
		else if (result == Rdex.Ternary.FALSE || (mCurrentFile == null && mContentUri == null)) {
			// decrypt file failed and we need to clean up or no file previously loaded
			mCardList.resetCardList();
			setTitle(R.string.app_name);
			mCurrentFile = null;
			mContentUri = null;
			showCard(getString(R.string.no_file_loaded));
		}
		// else decrypt file failed but previous state remains unchanged
	}

	// so decrypt dialog response can call back to store filename or uri
	public void updateFileLoadState(File myFile, Uri uri, boolean updateList) {
		if (myFile != null) {
			String filename = myFile.getAbsolutePath();
			updateRecentFileList(filename, true);
			mCurrentFile = filename;
			mContentUri = null;
			mDisplayName = "";
			setScreenTitle();
		}
		else if (uri != null) {
			if (updateList) {
				// don't want to update state for a content uri from an incoming intent
				updateRecentFileList(uri.toString(), true);
			}
			mCurrentFile = null;
			mContentUri = uri;
			mDisplayName = getUriDisplayName(uri);
			setScreenTitle();
		}
	}

	// called by gesture detector to select text and copy it to clipboard
    private void locateText(int x, int y) {
    	Layout layout = mBodyText.getLayout();
    	if (layout != null)
    	{
    	    int scrollY = mBodyText.getScrollY();
    	    int line = layout.getLineForVertical(scrollY + y);
    	    int offset = layout.getOffsetForHorizontal(line, x);
    	    CharSequence str = mBodyText.getText();
    	    if (offset >= str.length()) return;
    	    int end = offset;
    	    while (offset > 0 && !Character.isWhitespace(str.charAt(offset - 1)) &&
    	    		str.charAt(offset - 1) != '<') {
    	    	--offset;
    	    }
    	    while (end < str.length() && !Character.isWhitespace(str.charAt(end)) &&
    	    		str.charAt(end) != '>') {
    	    	++end;
    	    }
     	    if (str.charAt(end - 1) == '.') --end;
         	if (end <= offset) return;

         	if (mCopyIndex > -1 && offset > mCopyIndex) {
         		//copy entire section
         		offset = mCopyIndex;
             	showCard(str.toString(), offset, end);
         	} else {
         		//copy word
             	showCard(str.toString(), offset, end);
             	mCopyIndex = offset;
         	}

         	//copy selection to the clipboard
			final ClipboardManager clipboard = (ClipboardManager)
					getSystemService(Context.CLIPBOARD_SERVICE);
			final ClipData clipData = ClipData.newPlainText("plain-text", str.subSequence(offset, end));
			clipboard.setPrimaryClip(clipData);
       	}
    }

    private void displayFileOnOpening() {
    	if (mOpenInListView && mCardList.getNumCards() > 1 && mCardList.getNumCards() <= MAX_LIST_ALL_ITEMS) {
			mEditText.setText("");
			listAllCards("");
		} else {
			mCardList.showCard();
		}
	}

	public void showStatusMsg(String msg) {
    	Toast.makeText(Rdex.this, msg, Toast.LENGTH_LONG).show();
    }

    public void showCard(String cardText) {
    	mBodyText.scrollTo(0,0);
    	mBodyText.setText(cardText, TextView.BufferType.SPANNABLE);

    	if (mShowCardId) setScreenTitle();

    	//soft keyboard may need hiding
    	mInputMgr.hideSoftInputFromWindow(mBodyText.getWindowToken(), 0);
    	mPendingLayout = -1;
    	mCopyIndex = -1;
    }

    //show card and mark selected text
    public void showCard(String cardText, int begin, int end) {
    	mBodyText.scrollTo(0,0);
    	mBodyText.setText(cardText, TextView.BufferType.SPANNABLE);

    	Spannable WordtoSpan = (Spannable) mBodyText.getText();
    	WordtoSpan.setSpan(new BackgroundColorSpan(HIGHLIGHT_BACKGROUND_COLOR), begin, end,
    			Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
    	WordtoSpan.setSpan(new ForegroundColorSpan(HIGHLIGHT_FOREGROUND_COLOR), begin, end,
    			Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);

    	mBodyText.setText(WordtoSpan);
    	if (mLayoutFinished) {
    		mPendingLayout = -1;
    		if (begin > 0) {
    			//on some systems bringPointIntoView() fails to do anything when called
    			//immediately after setText() so set a timer in addition
	    		mPendingLayout = begin;
	            mHandler.removeCallbacks(mTimeoutCallback);
	            mHandler.postDelayed(mTimeoutCallback, 50);
	    		mBodyText.bringPointIntoView(begin);
    		}
    	}
    	else if (begin > 0) {
	        //bringPointIntoView() will crash if called before layout finished
    		//so do it on layout complete event
    		mPendingLayout = begin;
    	}

    	if (mShowCardId) setScreenTitle();

    	//need to hide soft keyboard because it was probably used to enter search string
    	mInputMgr.hideSoftInputFromWindow(mBodyText.getWindowToken(), 0);
    	mCopyIndex = -1;
    }

    private void setScreenTitle() {
		int cardId = mCardList.getCurrentCardId();
		String screenTitle = getString(R.string.app_name);
		String filename = null;
		if (mCurrentFile != null)
			filename = extractFileName(mCurrentFile);
		else if (mContentUri != null && mDisplayName != null && mDisplayName.length() > 0)
			filename = mDisplayName;
		if (filename == null) {
			if (mShowCardId && cardId > -1) {
				screenTitle += "  card: " + cardId;
			}
		}
		else if (mShowCardId && cardId > -1) {
			screenTitle += "  " + cardId + ":" + filename;
		} else {
			screenTitle += " -- " + filename;
		}
		setTitle(screenTitle);
    }

    public void alertMsg(String msg) {
    	new AlertDialog.Builder(this)
    		.setMessage(msg)
    		.setPositiveButton("OK",
    			new DialogInterface.OnClickListener() {
    				@Override
    				public void onClick(DialogInterface dialog, int which) { }
    			}).show();
    }


    //list first lines of all cards containing search string
    private void listAllCards(String searchString) {
    	ArrayList<String> titleList = new ArrayList<>();
    	mCardIdList.clear();
    	if (!mCardList.listCards(searchString, mCardIdList, titleList, mListTitlesAlpha)) {
			showStatusMsg("Too many cards in list");
			return;
		}
    	if (mCardIdList.size() == 0) {
			if (searchString.length() > 0) showStatusMsg("\"" + searchString + "\" not found");
			else showStatusMsg("Empty card list");
    		return;
    	}
    	if (mCardIdList.size() == 1) {
    		mCardList.setCurrentCardId(mCardIdList.get(0));
			mCardList.setCardIndex(-1);
    		if (searchString.length() > 0) {
    			mCardList.searchFwd(searchString);
    		} else {
    			mCardList.showCard();
    		}
    		return;
    	}

    	Intent i = new Intent(this, SearchList.class);
        i.putStringArrayListExtra(KEY_DISPLAY_LIST, titleList);
        String pageTitle = (searchString.length() > 0) ? getString(R.string.list_cards) : getString(R.string.list_all_cards);
        if (mCurrentFile != null && mCurrentFile.length() > 0)
        	pageTitle += " -- " + extractFileName(mCurrentFile);
        i.putExtra(KEY_DISPLAY_PAGE_TITLE, pageTitle);
        startActivityForResult(i, ACTIVITY_DISPLAY_LIST);
    }


    private void updateRecentFileList(String filename, boolean addFilename) {
    	if (filename == null) return;
		// create recent files list
    	ArrayList<String> fileList = new ArrayList<>();
    	if (addFilename) fileList.add(filename);	// add filename, else remove
		SharedPreferences settings = getPreferences(0);
		String str;
		try {
			boolean filenameRemoved = false;
			for (int i=0; i < MAX_RECENT_FILES; ++i) {
				str = settings.getString("filename" + i, null);
				if (str != null) {
					if (filename.compareTo(str) == 0) {
						filenameRemoved = true;
						continue;
					}
					fileList.add(str);
				}
			}

			//write out new list
			SharedPreferences.Editor editor = settings.edit();
			int endStop = Math.min(fileList.size(), MAX_RECENT_FILES);
			for (int i=0; i < endStop; ++i) {
				if (fileList.get(i) != null) {
					editor.putString("filename" + i, fileList.get(i));
				}
			}
            if (!addFilename && filenameRemoved) {
                // remove last entry
                editor.remove("filename" + endStop);
            }
			editor.apply();
		}
		catch (ClassCastException e) {
			//ignore exceptions
		}
    }

	//remove storage dir path from filename
	public String extractFilePath(String filename) {
		if (filename == null) return "";
		int index = filename.indexOf("/Android/data/");
		if (index > -1) {
			return filename.substring(index);
		}
		return filename;
	}

	//remove path from filename
	private String extractFileName(String filename) {
		if (filename == null) return "";
        int index = filename.lastIndexOf('/');
        if (index > -1 && index + 1 < filename.length()) {
        	return filename.substring(index + 1);
        }
        else return filename;
	}

	// the display name of a content uri is usually it's filename
	private String getUriDisplayName(Uri uri) {
		String displayName = "";
		try {
			Cursor cursor = getContentResolver().query(uri, null, null,
					null, null);
			if (cursor != null && cursor.moveToFirst()) {
				int nameIndex = cursor.getColumnIndex(OpenableColumns.DISPLAY_NAME);
				if (!cursor.isNull(nameIndex)) {
					displayName = cursor.getString(nameIndex);
				}
			}
			if (cursor != null) cursor.close();
		}
		// a query for a deleted file will issue a security exception
		catch (SecurityException e) { displayName = ""; }
		return displayName;
	}

	private String getFileDisplayName() {
		String filestr = "";
		if (mCurrentFile != null) filestr = extractFileName(mCurrentFile);
		else if (mContentUri != null) filestr = getUriDisplayName(mContentUri);
		if (filestr.length() > 0)
			filestr = "\"" + filestr + "\" ";
		return filestr;
	}

	public void openedFileStatusMsg() {
		String filestr;
		if (mCurrentFile != null) filestr = "Opened file: " + getFileDisplayName();
		else if (mContentUri != null) filestr = "Opened file: " + getFileDisplayName() + "from server";
		else return;
		showStatusMsg(filestr);
	}

	public void savedFileStatusMsg() {
		String filestr;
		if (mCurrentFile != null) filestr = "Saved file: " + getFileDisplayName();
		else if (mContentUri != null) filestr = "Saved file: " + getFileDisplayName() + "to server";
		else return;
		showStatusMsg(filestr);
	}

	private void newCard() {
		if (mReadOnly) {
			alertMsg(getString(R.string.read_only_alert));
			return;
		}
		if (mCurrentFile == null && mContentUri == null) {
			alertMsg(getString(R.string.no_file_specified));
			return;
		}
        float textSize = mBodyText.getTextSize();
		Intent i = new Intent(this, EditCard.class);
        i.putExtra(KEY_CARD_TEXT, "");
        i.putExtra(KEY_CARD_TEXT_SIZE, textSize);
        i.putExtra(KEY_DISPLAY_PAGE_TITLE, getString(R.string.new_card_title));
        startActivityForResult(i, ACTIVITY_NEW_CARD);
	}


	private void editCard() {
		if (checkCardlistIssues()) return;
		String cardText = mCardList.getCardText();
		if (cardText == null) return;

        float textSize = mBodyText.getTextSize();
		Intent i = new Intent(this, EditCard.class);
        i.putExtra(KEY_CARD_TEXT, cardText);
        i.putExtra(KEY_CARD_TEXT_SIZE, textSize);
        i.putExtra(KEY_DISPLAY_PAGE_TITLE, getString(R.string.edit_card_title));
        startActivityForResult(i, ACTIVITY_EDIT_CARD);
	}

	private void deleteCard() {
        if (checkCardlistIssues()) return;
		new AlertDialog.Builder(this)
//		.setIcon(R.drawable.icon)
		.setMessage("Are you sure you want to delete this card?")
		.setPositiveButton("Delete", new DialogInterface.OnClickListener() {
				@Override
				public void onClick(DialogInterface dialog, int which) {
					//yes, delete the card
			    	if (mCardList.deleteCard()) saveFile();
				}
			})
		.setNegativeButton("Cancel", new DialogInterface.OnClickListener() {
					@Override
					public void onClick(DialogInterface dialog, int which) { }
				})
		.show();
	}

    private boolean checkCardlistIssues() {
        if (mReadOnly) {
            alertMsg(getString(R.string.read_only_alert));
            return true;
        }
        if (mCardList.getNumCards() < 1) {
            alertMsg(getString(R.string.cardlist_empty_alert));
            return true;
        }
        if (mCardList.getCurrentCardId() < 0) {
            alertMsg(getString(R.string.no_card_selected_alert));
            return true;
        }
        return false;
    }

    private void openRecentFileDialog() {
    	mRecentFileList.clear();
    	ArrayList<String> recentFilenameList = new ArrayList<>();
		SharedPreferences settings = getPreferences(0);
    	for (int i=0; i < MAX_RECENT_FILES; ++i) {
			// create list of recent files, fileStr may be a filename or a uri
			String fileStr = settings.getString("filename" + i, null);
			String filename = getFileName(fileStr);
			if (filename != null && filename.length() > 0) {
				mRecentFileList.add(fileStr);
				recentFilenameList.add(filename);
			}
		}
    	if (mRecentFileList.size() == 0) {
			showStatusMsg("Recent file list empty.");
    		return;
		}

    	Intent i = new Intent(this, SearchList.class);
        i.putStringArrayListExtra(KEY_DISPLAY_LIST, recentFilenameList);
        i.putExtra(KEY_DISPLAY_PAGE_TITLE, getString(R.string.menu_recent_files));
        startActivityForResult(i, ACTIVITY_RECENT_FILE_LIST);
    }

	private boolean isContentUrl(String fileStr) {
		if (fileStr != null && fileStr.length() > 0) {
			Uri uri = Uri.parse(fileStr);
			if (uri != null) {
				String scheme = uri.getScheme();
				return (scheme != null && scheme.equals("content"));
			}
		}
		return false;
	}

	private String getFileName(String fileStr) {
		if (fileStr != null && fileStr.length() > 0) {
			String filename = null;
			Uri uri = Uri.parse(fileStr);
			String scheme = null;
			if (uri != null)
				scheme = uri.getScheme();
			if (scheme == null) {
				filename = fileStr;
			} else if (scheme.equals("file")) {
				filename = uri.getEncodedPath();
			} else if (scheme.equals("content")) {
				return getUriDisplayName(uri);
			}
			if (filename != null && filename.length() > 0) {
				//separate filename from path
				int index = filename.lastIndexOf('/');
				if (index > -1 && index + 1 < filename.length())
					return filename.substring(index + 1);
				else
					return filename;
			}
		}
		return null;
	}

    //initiate open file activity
    private void openFileDialog() {
    	String state = Environment.getExternalStorageState();
    	if (!Environment.MEDIA_MOUNTED.equals(state) &&
    		!Environment.MEDIA_MOUNTED_READ_ONLY.equals(state)) {
    			alertMsg("Media is unavailable");
        		return;
    	}
        Intent i = new Intent(this, FileDialog.class);
        startActivityForResult(i, ACTIVITY_GET_FILENAME);
    }

	private void openSafFile() {
		Intent intent = new Intent(Intent.ACTION_OPEN_DOCUMENT);
		intent.addCategory(Intent.CATEGORY_OPENABLE);
		intent.setType("*/*");

		startActivityForResult(intent, ACTIVITY_OPEN_SAF_FILE);
	}

	// Create new saf file and save as saf file
	private void createSafFile(int activityId) {
    	Intent intent = new Intent(Intent.ACTION_CREATE_DOCUMENT);
		intent.addCategory(Intent.CATEGORY_OPENABLE);
		intent.setType("application/octet-stream");

		startActivityForResult(intent, activityId);
	}

	private Uri getUriFromIntent(Intent intent) {
		if (intent != null) {
			Uri uri = intent.getData();
			if (uri != null) {
				String scheme = uri.getScheme();
				if (scheme != null && scheme.equals("content"))
					return uri;
			}
		}
		return null;
	}

	@Override
    protected void onActivityResult(int requestCode, int resultCode, Intent intent) {
        super.onActivityResult(requestCode, resultCode, intent);
		//result of create new SAF file
		if (requestCode == ACTIVITY_CREATE_SAF_FILE && resultCode == RESULT_OK) {
			Uri uri = getUriFromIntent(intent);
			if (uri != null) {
				// create file and reset cardlist
				mCurrentFile = null;
				mContentUri = uri;
				mDisplayName = getUriDisplayName(uri);
				takePersistableUriPermission(uri, intent);
				mCopyIndex = -1;
				mCrypto.deleteCurrentPasswd();
				mCrypto.resetUseDefaultPasswd();
				mFileEncoding = RdxEncoding.RDX_UTF8;
				//default encoding for new files is utf-8
				mCardList.resetCardList();
				showCard(getString(R.string.new_file_help));
				updateRecentFileList(uri.toString(), true);
				setScreenTitle();
				if (mDisplayName != null)
					showStatusMsg("New cardfile \"" + mDisplayName + "\" created");
			}
		}
		//result of open SAF file picker
		if (requestCode == ACTIVITY_OPEN_SAF_FILE && resultCode == RESULT_OK) {
			Uri uri = getUriFromIntent(intent);
			if (uri != null) {
				Ternary result = loadContentFromUri(uri);
				if (result == Ternary.TRUE || result == Ternary.NEUTRAL) {
					// Ternary NEUTRAL indicates waiting for password dialog
					takePersistableUriPermission(uri, intent);
				}
				if (result == Ternary.TRUE) {
					displayFileOnOpening();
					openedFileStatusMsg();
				}
			}
		}
		//result of save as SAF file picker
		if (requestCode == ACTIVITY_SAVE_AS_SAF_FILE && resultCode == RESULT_OK) {
			Uri uri = getUriFromIntent(intent);
			if (uri != null) {
				saveLocalCopy(uri);
			}
		}
        //result of legacy open file dialog
        if (requestCode == ACTIVITY_GET_FILENAME && resultCode == RESULT_OK) {
        	String filename = intent.getStringExtra(KEY_FILE_NAME);
        	if (filename != null && filename.length() > 0) {
				// Ternary NEUTRAL indicates waiting for password dialog
	        	if (loadFile(filename, false) == Ternary.TRUE) {
					displayFileOnOpening();
					openedFileStatusMsg();
	        	}
        	}
        }
        //result of open recent files
        else if (requestCode == ACTIVITY_RECENT_FILE_LIST && resultCode == RESULT_OK) {
        	int result = intent.getIntExtra(KEY_LIST_SELECTION, -1);
        	if (result > -1 && mRecentFileList.size() > 0) {
        		String fileStr = mRecentFileList.get(result);
            	if (fileStr != null && fileStr.length() > 0) {
                    Ternary success = loadFileNameOrUri(fileStr);
    	        	if (success == Ternary.TRUE) {
						displayFileOnOpening();
						openedFileStatusMsg();
    	        	}
                    else if (success == Ternary.FALSE) {
                        // remove from recent file list if present
                        updateRecentFileList(fileStr, false);
                    }
            	}
        	}
        }
        //result of list first lines
        else if (requestCode == ACTIVITY_DISPLAY_LIST && resultCode == RESULT_OK) {
        	int result = intent.getIntExtra(KEY_LIST_SELECTION, -1);
        	if (result > -1 && mCardIdList.size() > 0) {
        		mCardList.setCurrentCardId(mCardIdList.get(result));
    			mCardList.setCardIndex(-1);
        		if (mEditText.getText().toString().length() > 0) {
        			mCardList.searchFwd(mEditText.getText().toString());
        		} else {
        			mCardList.showCard();
        		}
        		return;
        	}
			//handle menu item selection in list all screen
        	result = intent.getIntExtra(KEY_MENU_SELECTION, -1);
        	if (result > -1) {
        		if (!handleMenuItem(result)) {
        			showStatusMsg("Bad menu item id.");
        		}
        	}
        }
        //result of set preferences
        else if (requestCode == ACTIVITY_SET_PREFERENCES) {
        	//text display preferences may have changed
        	restorePreferences();
            if (mCardList.getNumCards() > 0) mCardList.showCard();
        }
        //result of new card
        else if (requestCode == ACTIVITY_NEW_CARD && resultCode == RESULT_OK) {
        	String cardText = intent.getStringExtra(Rdex.KEY_CARD_TEXT);
        	if (cardText != null) {
        		mCardList.addNewCard(cardText);
        		showCard(cardText);
        		saveFile();
        	}
        }
        //result of edit card
        else if (requestCode == ACTIVITY_EDIT_CARD && resultCode == RESULT_OK) {
        	String cardText = intent.getStringExtra(Rdex.KEY_CARD_TEXT);
        	if (cardText != null) {
        		mCardList.updateCurrentCard(cardText);
        		showCard(cardText);
        		saveFile();
        	}
        }
    }

    // preserve access to files from content uri across device restarts
    private void takePersistableUriPermission(Uri uri, Intent intent) {
		final int takeFlags = intent.getFlags()
				& (Intent.FLAG_GRANT_READ_URI_PERMISSION | Intent.FLAG_GRANT_WRITE_URI_PERMISSION);
		getContentResolver().takePersistableUriPermission(uri, takeFlags);
	}

	private boolean checkFileExists(Uri uri) {
		if (uri == null) return false;
		boolean result = false;
		try {
			Cursor cursor = getContentResolver().query(uri, null, null,
					null, null);
			if (cursor != null) {
				result = cursor.moveToFirst();
				cursor.close();
			}
		}
		// a query for a deleted file will issue a security exception
		catch (SecurityException e) { result = false; }
    	return result;
	}

    private Ternary loadFileNameOrUri(String fileStr) {
		if (fileStr == null || fileStr.length() == 0)
			return Ternary.FALSE;

		Uri uri = Uri.parse(fileStr);
		String scheme = null;
		if (uri != null)
			scheme = uri.getScheme();
		if (scheme == null) {
			return loadFile(fileStr, false);
		}
		else if (scheme.equals("file")) {
			String fileName = uri.getEncodedPath();
			if (fileName != null && fileName.length() > 0)
				return loadFile(fileName, false);
		}
		else if (scheme.equals("content")) {
			return loadContentFromUri(uri);
		}
		return Ternary.FALSE;
	}

	// load file given file path
    private Ternary loadFile(String filename, boolean noDialog) {
		if (filename == null || filename.length() == 0) return Ternary.FALSE;

		String state = Environment.getExternalStorageState();
		if (!Environment.MEDIA_MOUNTED.equals(state) &&
				!Environment.MEDIA_MOUNTED_READ_ONLY.equals(state)) {
			alertMsg("Media is unavailable");
			return Ternary.FALSE;
		}

		File myFile = new File(filename);
        String errorStr = "File \"" + extractFilePath(filename) + "\" ";
		Ternary result = loadFileOrContent(myFile, null, errorStr, noDialog, false);
		if (result == Ternary.TRUE) {
			if (mFileEncoding == RdxEncoding.RDX_UTF8
					|| mFileEncoding == RdxEncoding.RDX_ASCII
					|| mFileEncoding == RdxEncoding.RDX_AES128) {
				//only update recent file list for rdx files
				updateRecentFileList(filename, true);
			}
			mCurrentFile = filename;
			mContentUri = null;
			mDisplayName = "";
			setScreenTitle();
		}
		return result;
	}

	private Ternary loadContentFromUri(Uri uri) {
		return loadContentFromUri(uri, false, true);
	}

	// load file given content uri
	// updateList prevents us from adding a content uri from an incoming intent at startup
	// to the recent files list (for a filename it doesn't matter but a uri may have come
	// from a content server and we don't want to save it)
	private Ternary loadContentFromUri(Uri uri, boolean noDialog, boolean updateList) {
		if (!checkFileExists(uri)) return Ternary.FALSE;
		String displayName = getUriDisplayName(uri);
		if (displayName.length() > 0)
			displayName = "\"" + displayName + "\" ";
		String errorStr = "File " + displayName + "from content URI ";
		Ternary result = loadFileOrContent(null, uri, errorStr, noDialog, updateList);
		if (result == Ternary.TRUE) {
			if (updateList && (mFileEncoding == RdxEncoding.RDX_UTF8
					|| mFileEncoding == RdxEncoding.RDX_ASCII
					|| mFileEncoding == RdxEncoding.RDX_AES128)) {
				//only update recent file list for rdx files
				updateRecentFileList(uri.toString(), true);
			}
			mCurrentFile = null;
			mContentUri = uri;
			mDisplayName = getUriDisplayName(uri);
			setScreenTitle();
		}
		return result;
	}

	// noDialog indicates crypto state has been restored via save instance state
	// updateList prevents us from adding a content uri from an incoming intent at startup
	// to the recent files list if we need to wait for the user to enter a passwd
	// (neutral result indicates file operation pending while waiting for user input)
	private Ternary loadFileOrContent(File myFile, Uri uri, String errorStr,
									  boolean noDialog, boolean updateList) {
		final int BUFSIZE = 4096;

		if (myFile == null && uri == null) return Ternary.FALSE;
		char[] buf = new char[BUFSIZE];
    	int bytesRead;
    	boolean utf8 = false;
		boolean rdex_utf8 = false;
		boolean aes128 = false;
		mCopyIndex = -1;

    	//check for rdex utf-8 or aes-128 headers
    	try {
            InputStream inStream;
			if (myFile != null) inStream = new FileInputStream(myFile);
			else inStream = getContentResolver().openInputStream(uri);
            InputStreamReader rdr = new InputStreamReader(inStream, "8859_1");
                //8859_1 and utf8 are the only encodings guaranteed present
    		int hdrLen = RDEX_FILE_UTF8_FORMAT_HDR.length();
    		final int UTF8_BOM_LEN = 3;
    		bytesRead = rdr.read(buf, 0, hdrLen + UTF8_BOM_LEN);
    		rdr.close();
    		if (bytesRead >= UTF8_BOM_LEN) {
    			String hdr = new String(buf, 0, bytesRead);
				if (hdr.startsWith(UTF8_BYTE_ORDER_MARK)) utf8 = true;
					//permit utf-8 plain text file
    			if (hdr.startsWith(RDEX_FILE_UTF8_FORMAT_HDR)) { utf8 = true; rdex_utf8 = true; }
    			else if (utf8 && hdr.startsWith(RDEX_FILE_UTF8_FORMAT_HDR, 3)) rdex_utf8 = true;
    			    //permit utf-8 byte order mark in rdex header
				else if (hdr.startsWith(RDEX_FILE_AES128_FORMAT_HDR)) aes128 = true;
                else if (hdr.startsWith(RDEX_FILE_UNRECOGNIZED_HDR_01) ||
                        hdr.startsWith(RDEX_FILE_UNRECOGNIZED_HDR_02)) {
                    alertMsg(errorStr + getString(R.string.unrecognized_rdex_header));
                    return Ternary.FALSE;
                }
    		}
		} catch (FileNotFoundException e) {
			alertMsg(errorStr + "not found");
			return Ternary.FALSE;
		} catch (UnsupportedEncodingException e) {
			alertMsg(errorStr + "file encoding error");
			return Ternary.FALSE;
		} catch (IOException e) {
    		alertMsg(errorStr + "file header read error");
			return Ternary.FALSE;
		}

		if (aes128) {
			// load encrypted file
			// noDialog indicates crypto state has been restored via save instance state
			Ternary result = mCrypto.decryptCardlist(myFile, uri, errorStr, noDialog, updateList);
			if (result == Ternary.TRUE) {
				mFileEncoding = RdxEncoding.RDX_AES128;
			}
			return result;
		}

		InputStreamReader rdr;
    	StringBuilder stringBuf = new StringBuilder();
    	int numCards = 0;
    	bytesRead = 0;
    	boolean skipFirstCard = rdex_utf8;
    	try {
			InputStream inStream;
            if (myFile != null) inStream = new FileInputStream(myFile);
            else inStream = getContentResolver().openInputStream(uri);
            if (utf8) rdr = new InputStreamReader(inStream, "UTF8");
            else rdr = new InputStreamReader(inStream);
			while (bytesRead > -1) {
		    	bytesRead = rdr.read(buf, 0, BUFSIZE);
		    	for (int i=0; i < bytesRead; ++i) {
		    		if (buf[i] == CardList.SEPARATOR && (rdex_utf8 || !utf8)) {
                        // (ignore card separator for utf-8 files without a valid utf-8 rdex header)
		    			if (numCards == 0) {
		    				if (skipFirstCard) {
		    					//skip rdex utf-8 header
		    					skipFirstCard = false;
		    					stringBuf.setLength(0);
		    					continue;
		    				}
		    				mCardList.resetCardList();
		    			}
		    			mCardList.addCard(stringBuf.toString());
		    			stringBuf.setLength(0);
		    			++numCards;
		    		}
		    		else if (buf[i] != '\r') stringBuf.append(buf[i]);
		    	}
	    	}
			if (numCards == 0) {
				if (stringBuf.length() > 0 && !rdex_utf8) {
					//no card separators in file, rdx files always end with a separator
					//maybe it was a txt file, add file as a single card
					mCardList.resetCardList();
					mCardList.addCard(stringBuf.toString());
					if (utf8) mFileEncoding = RdxEncoding.TXT_UTF8;
					else  mFileEncoding = RdxEncoding.TXT_ASCII;
					mCrypto.deleteCurrentPasswd();
					mCrypto.resetUseDefaultPasswd();
				} else {
                    if (stringBuf.length() == 0) alertMsg(errorStr + "is empty");
                    else alertMsg(errorStr + "contains no valid cards");
					rdr.close();
					return Ternary.FALSE;
				}
			} else {
                if (!utf8) mFileEncoding = RdxEncoding.RDX_ASCII;
				else  mFileEncoding = RdxEncoding.RDX_UTF8;
				mCrypto.deleteCurrentPasswd();
				mCrypto.resetUseDefaultPasswd();
			}
			rdr.close();
    	} catch (IOException e) {
	    		alertMsg("file read exception");
	    		mCardList.resetCardList();
				setTitle(R.string.app_name);
				return Ternary.FALSE;
	    }
    	return Ternary.TRUE;
    }


	private void displayAbout() {
		String versionName = " version ";
		try {
			versionName += getPackageManager().getPackageInfo(getPackageName(), 0).versionName;
		} catch (PackageManager.NameNotFoundException e) {
			versionName = "";
		}
		String aboutInfo = "Rdex" + versionName;
		aboutInfo += "\nCopyright 2023 Peter Newman\n";
		aboutInfo += "www.pnewman.com/rdex\n";
		aboutInfo += "Bug reports to <rdex@pnewman.com>\n\n";

		if (mCurrentFile != null || mContentUri != null) {
			if (mCurrentFile != null) {
				int index = mCurrentFile.lastIndexOf('/');
				if (index > -1 && index + 1 < mCurrentFile.length()) {
					aboutInfo += "Filename: " + mCurrentFile.substring(index + 1);
					aboutInfo += "\nLocation: " + mCurrentFile.substring(0, index + 1);
				} else {
					aboutInfo += "Filename: " + mCurrentFile;
				}
			} else {
				if (mDisplayName != null && mDisplayName.length() > 0)
					aboutInfo += "Filename: " + mDisplayName + "\n";
				aboutInfo += "File from server: " + mContentUri.getAuthority();
			}
			if (mFileEncoding == RdxEncoding.RDX_AES128)
				aboutInfo += "\n(AES-128 Encrypted Format)";
			else if (mFileEncoding == RdxEncoding.RDX_UTF8 || mFileEncoding == RdxEncoding.TXT_UTF8)
				aboutInfo += "\n(UTF-8 Format)";
			else if (mFileEncoding == RdxEncoding.RDX_ASCII || mFileEncoding == RdxEncoding.TXT_ASCII)
				aboutInfo += "\n(ASCII Format)";
		} else {
			aboutInfo += "No file loaded";
		}
		
		aboutInfo += "\nNumber of cards: " + mCardList.getNumCards() + "\n\n";

		aboutInfo +=
		"This program is free software; you can redistribute it and/or modify it under " +
		"the terms of the GNU General Public License as published by the Free Software " +
		"Foundation; either version 3 of the License, or (at your option) any later version.\n\n";

		aboutInfo +=
		"This program is distributed in the hope that it will be useful, but WITHOUT ANY " +
		"WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A " +
		"PARTICULAR PURPOSE.  See the GNU General Public License for more details: " +
		"www.gnu.org/licenses/.";
			
		Intent i = new Intent(this, Help.class);
        i.putExtra(KEY_DISPLAY_HELP_FILE, aboutInfo);
        i.putExtra(KEY_DISPLAY_PAGE_TITLE, getString(R.string.about_page_title));
        startActivity(i);
	}
	
	
    private void loadHelpFile() {
    	final int BUFSIZE = 1024;

    	char[] buf = new char[BUFSIZE];
    	InputStream is = getResources().openRawResource(R.raw.helpfile);
    	InputStreamReader isr = new InputStreamReader(is);
    	StringBuilder stringBuf = new StringBuilder();
		int bytesRead = 0;
    	try {
			while (bytesRead > -1) {
    			bytesRead = isr.read(buf, 0, BUFSIZE);
		    	for (int i=0; i < bytesRead; ++i) {
		    		if (buf[i] != '\r') stringBuf.append(buf[i]);
		    	}
			}
			isr.close();
    	}
    	catch (IOException e) {
    		alertMsg("File Read Exception reading help file");
    		return;
    	}

    	if (stringBuf.length() == 0) {
    		alertMsg("Problem reading help file");
    		return;
    	}
    	
    	Intent i = new Intent(this, Help.class);
        i.putExtra(KEY_DISPLAY_HELP_FILE, stringBuf.toString());
        i.putExtra(KEY_DISPLAY_PAGE_TITLE, getString(R.string.help_dialog_title));
        startActivity(i);
    }

    // from new file dialog or save as dialog: check filename and create new file if necessary
	// then reset cardlist for new file or save local file for save as
	private void checkFilename(String filename, int dialogType) {
		final String[] ReservedChars =
			{"|", "\\", "?", "*", "<", "\"", ":", ">", "/", ";", "%", "@"};

		for (String c : ReservedChars) {
			if (filename.contains(c)) {
				String errorStr = "The character \"" + c + "\" is invalid for use in a filename";
				badNewFileMsg(errorStr, dialogType);
				return;
			}
		}
		int index = filename.indexOf(".rdx");
		if (index == -1 || (filename.length() > 4 && index < filename.length() - 4)) {
			filename += ".rdx";
		}
		if (dialogType == DIALOG_SAVE_AS_ID) {
			if (myFileExists(filename)) {
				// ask user whether to overwrite file
				saveAsOverwriteFileDialog(filename);
			}
			else if (createNewFile(filename, false)) {
				saveLocalCopy(filename);
			}
		} else {
			// create new private cardfile, now disabled
			if (myFileExists(filename)) {
				badNewFileMsg("\"" + filename + "\" file already exists", dialogType);
				return;
			}
			if (createNewFile(filename, true)) {
				// filename okay, create file and reset cardlist
				mCardList.resetCardList();
				mCopyIndex = -1;
				showCard(getString(R.string.new_file_help));
				updateRecentFileList(mCurrentFile, true);
				setTitle(getString(R.string.app_name) + " -- " + extractFileName(filename));
				showStatusMsg("New cardfile \"" + filename + "\" created");
			}
		}
	}

    private void badNewFileMsg(String msg, final int dialogType) {
    	new AlertDialog.Builder(this)
    		.setMessage(msg)
    		.setPositiveButton("OK", 
    			new DialogInterface.OnClickListener() {
    				@Override
    				public void onClick(DialogInterface dialog, int which) {
    					//bad filename, try again
						showDialogFragment(dialogType);
    				}
    			}).show();
    }

	// dialog to ask whether to overwrite file in save as
	private void saveAsOverwriteFileDialog(final String filename) {
		new AlertDialog.Builder(this)
				.setTitle(R.string.save_as_overwrite_dialog_title)
				.setMessage("The file \"" + filename + "\" already exists. Do you want to overwrite it?")
				.setPositiveButton("Yes", new DialogInterface.OnClickListener() {
					@Override
					public void onClick(DialogInterface dialog, int id) {
						saveLocalCopy(filename);
					}
				})
				.setNegativeButton("No", new DialogInterface.OnClickListener() {
					@Override
					public void onClick(DialogInterface dialog, int id) { showDialogFragment(DIALOG_SAVE_AS_ID); }
				})
				.show();
	}

    private String getExtDirName() {
		File extDir = getExternalFilesDir(null);
		if (extDir == null) return null;
		else return extDir.getAbsolutePath();
    }

    //test if file exists
    private boolean myFileExists(String filename) {
    	String dirname = getExtDirName();
    	if (dirname == null) return false;
	    File myFile = new File(dirname);
	    if (!myFile.exists()) {
	    	//directory path does not exist
			return false;
	    }
		String pathname = dirname + File.separator + filename;
		myFile = new File(pathname);
		//file does not exist
		return myFile.exists();
	}
    
    
    private boolean createNewFile(String filename, boolean useNewFile) {
    	String dirname = getExtDirName();
    	if (dirname == null) {
			alertMsg("Failed to access local storage.");
			return false;
		}
	    File myFile = new File(dirname);
		if (!myFile.exists()) {
			//create all missing components of directory path
	    	if (!myFile.mkdirs()) {
	 			alertMsg("Failed to create new folder \"" + dirname + "\"");
	 			return false;
	        }
		}
		filename = dirname + File.separator + filename;
		myFile = new File(filename);
		try {
			if (!myFile.exists()) {
				//create new cardfile
		    	if (!myFile.createNewFile()) {
		 			alertMsg("Failed to create new file \"" + filename + "\"");
		 			return false;
		        }
			} else {
	 			alertMsg("Failed to create new file \"" + filename + "\" file already exists");
	 			return false;
			}
		} catch (IOException e) {
			alertMsg("File \"" + filename + "\" Failed to create new file, probably a permissions issue");
				return false;
		}
		if (useNewFile) {
			// for new cardfile or new file from save but not for save as
			mCrypto.deleteCurrentPasswd();
			mCrypto.resetUseDefaultPasswd();
			mCurrentFile = filename;
			mFileEncoding = RdxEncoding.RDX_UTF8;
			//default encoding for new files is utf-8
		}
		return true;
    }

	public void saveFile() {
    	if (mContentUri == null) {
			String state = Environment.getExternalStorageState();
			if (!Environment.MEDIA_MOUNTED.equals(state)) {
				alertMsg("Save file failed: The media is not writable. Any previous copy of the file has not been modified.");
				return;
			}
		}

		boolean noFinalSeparator = (mFileEncoding == RdxEncoding.TXT_ASCII || mFileEncoding == RdxEncoding.TXT_UTF8);
    	String data = mCardList.cardListToString(noFinalSeparator);
    	if (data == null) {
			alertMsg("Save file failed: The cardfile is empty.");
			return;
		}
    	
    	boolean newFile = false;
    	if (mCurrentFile == null && mContentUri == null) {
    		// create new file
			// (it should no longer be possible to reach here without a file or content uri
			// Edit/New Card now requires a file or content uri, but the code can stay)
			mFileEncoding = RdxEncoding.RDX_UTF8;
    		if (!createNewFile("database.rdx", true)) return;
			newFile = true;
			alertMsg("You have created a new cardfile. It will be stored in UTF-8 format as database.rdx " +
					"in the Android/data/com.pnewman.rdex/files folder. You can leave it there or you can " +
					"copy it, rename it, upload it or move it using a file browser and then reopen it " +
					"at its new location using the Rdex \"Open File\" menu. " +
					"You can also encrypt it using the Rdex \"Encrypt\" menu.");
    	}
		else if (mFileEncoding == RdxEncoding.RDX_NONE) {
			// (I don't think it's possible to get here any more either)
			alertMsg("Rdex supports two plain text file formats, ASCII and UTF-8, " +
					"and one encrypted file format, AES-128. " +
					"The default is the UTF-8 format. This guarantees that characters do not " +
					"get corrupted when shared across different computer systems. The UTF-8 format " +
					"is not compatible with versions of Rdex " +
					"earlier than 1.5 in the 1.x series or 2.4.7 in the 2.x series.");
			mFileEncoding = RdxEncoding.RDX_UTF8;
		}

		if (mFileEncoding == RdxEncoding.RDX_AES128) {
			//save encrypted file
			Rdex.Ternary success = mCrypto.encryptFile(mCurrentFile, mContentUri, data);
			if (success == Rdex.Ternary.TRUE) savedFileStatusMsg();
			else if (success == Rdex.Ternary.NEUTRAL) {
				alertMsg("Encrypt and save operation failed. " +
						"However, any previous copy of the file has not been modified.");
			} else {
				alertMsg("Encrypt and save operation failed. " + getString(R.string.file_corrupted_error_msg));
			}
			return;
		}

		OutputStreamWriter wtr = null;
		try {
			OutputStream outStream;
			if (mCurrentFile != null) {
				File myFile = new File(mCurrentFile);
				outStream = new FileOutputStream(myFile);
			}
			else if (mContentUri != null) {
				// need to specifically truncate the file for api 29 by opening with mode 'wt'
				ParcelFileDescriptor pfd = getContentResolver().openFileDescriptor(mContentUri, "wt");
				if (pfd != null) {
					outStream = new FileOutputStream(pfd.getFileDescriptor());
				} else {
					alertMsg("Save file failed: Content server resource not available. " +
							"Any previous copy of the file has not been modified.");
					return;
				}
			} else {
				alertMsg("Save file failed: No filename or content server was specified. " +
						"Any previous copy of the file has not been modified.");
				return;
			}

    		if (mFileEncoding == RdxEncoding.RDX_ASCII || mFileEncoding == RdxEncoding.TXT_ASCII) {
				//default encoding
				wtr = new OutputStreamWriter(outStream);
			}
			else if (mFileEncoding == RdxEncoding.RDX_UTF8 || mFileEncoding == RdxEncoding.TXT_UTF8) {
				//utf-8 encoding
				wtr = new OutputStreamWriter(outStream, "UTF8");
			}
			if (wtr == null) {
				alertMsg("Save file failed. Any previous copy of the file has not been modified.");
				return;
			}

			if (mFileEncoding == RdxEncoding.RDX_UTF8) {
    			//write rdex utf-8 header
    			wtr.write(RDEX_FILE_UTF8_FORMAT_HDR, 0, RDEX_FILE_UTF8_FORMAT_HDR.length());
    			wtr.write(CardList.SEPARATOR);
    		}

			//save file
			wtr.write(data, 0, data.length());
			wtr.close();
			wtr = null;
			savedFileStatusMsg();
    		if (newFile) {
    			updateRecentFileList(mCurrentFile, true);
    			setTitle(getString(R.string.app_name) + " -- " + "database.rdx");
    		}
    	} catch (FileNotFoundException e) {
			alertMsg("Save file failed: specified file not found. " + getString(R.string.file_corrupted_error_msg));
		} catch (IOException e) {
 			alertMsg("Save file failed: write exception. " + getString(R.string.file_corrupted_error_msg));
 		} finally {
			try {
				if (wtr != null) wtr.close();
			} catch (IOException e) {
				// just ignore it
			}
		}
    }

	public void saveLocalCopy(String filename) {
		if (filename == null || filename.length() == 0) {
			alertMsg("Save private file copy failed: No filename specified.");
			return;
		}
		String state = Environment.getExternalStorageState();
		if (!Environment.MEDIA_MOUNTED.equals(state)) {
			alertMsg("Save private file copy failed: The media is not writable.");
			return;
		}
		String dirname = getExtDirName();
		if (dirname == null) {
			alertMsg("Save private file copy failed: Failed to access local storage.");
			return;
		}
		File dirFile = new File(dirname);
		if (!dirFile.exists()) {
			// this should not be possible -- file is known to exist, so its path must exist
			alertMsg("Save private file copy failed: failed to locate path to folder: \"" + dirname + "\"");
			return;
		}
		String pathname = dirname + File.separator + filename;
		writeLocalFileCopy(pathname, null, filename);
	}

	public void saveLocalCopy(Uri uri) {
		String displayName = getUriDisplayName(uri);
		writeLocalFileCopy(null, uri, displayName);
	}

	private void writeLocalFileCopy(String pathname, Uri uri, String displayName) {
		boolean noFinalSeparator = (mFileEncoding == RdxEncoding.TXT_ASCII || mFileEncoding == RdxEncoding.TXT_UTF8);
		String data = mCardList.cardListToString(noFinalSeparator);
		if (data == null) {
			alertMsg("Save copy failed: The cardfile is empty.");
			return;
		}

		OutputStreamWriter wtr = null;
		try {
			OutputStream outStream;
			if (pathname != null) {
				File myFile = new File(pathname);
				outStream = new FileOutputStream(myFile);
			}
			else if (uri != null) {
				// need to specifically truncate the file for api 29 by opening with mode 'wt'
				ParcelFileDescriptor pfd = getContentResolver().openFileDescriptor(uri, "wt");
				if (pfd != null) {
					outStream = new FileOutputStream(pfd.getFileDescriptor());
				} else {
					alertMsg("Save copy failed: Content server resource not available. " +
							"Please try to save the file again.");
					return;
				}
			} else {
				alertMsg("Save copy failed: No filename or content server was specified. " +
						"Please try to save the file again.");
				return;
			}

			if (mFileEncoding == RdxEncoding.RDX_ASCII || mFileEncoding == RdxEncoding.TXT_ASCII) {
				//default encoding
				wtr = new OutputStreamWriter(outStream);
			} else {
				//utf-8 encoding
				wtr = new OutputStreamWriter(outStream, "UTF8");
			}

			if (mFileEncoding == RdxEncoding.RDX_UTF8 || mFileEncoding == RdxEncoding.RDX_AES128) {
				//write rdex utf-8 header
				wtr.write(RDEX_FILE_UTF8_FORMAT_HDR, 0, RDEX_FILE_UTF8_FORMAT_HDR.length());
				wtr.write(CardList.SEPARATOR);
			}

			//save file
			wtr.write(data, 0, data.length());
			wtr.close();
			wtr = null;

			//saved copy msg
			String filestr = "Saved copy as: \"" + displayName + "\"\n";
			if (mFileEncoding == RdxEncoding.RDX_AES128) filestr += "(UTF-8 Format)";
			else if (mFileEncoding == RdxEncoding.RDX_UTF8) filestr += "(UTF-8 Format)";
			else if (mFileEncoding == RdxEncoding.RDX_ASCII) filestr += "(ASCII Format)";
			showStatusMsg(filestr);

		} catch (FileNotFoundException e) {
			alertMsg("Save copy failed: specified file \"" + pathname + "\" not found. " +
					"Please try to save the file again.");
		} catch (IOException e) {
			alertMsg("Save copy failed: write exception. Please try to save the file again.");
		} finally {
			try {
				if (wtr != null) wtr.close();
			} catch (IOException e) {
				// just ignore it
			}
		}
	}
}
