/*
Rdex version 2.4 for Android

Copyright (C) 2018 Peter Newman <pn@pnewman.com>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE.  See the GNU General Public License for more details:
<http://www.gnu.org/licenses/>.
*/

package com.pnewman.rdex;


import android.app.AlertDialog;
import android.content.DialogInterface;
import android.content.SharedPreferences;
import android.database.Cursor;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.provider.OpenableColumns;
import android.text.Editable;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.security.AlgorithmParameters;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidParameterSpecException;
import java.util.Arrays;
import java.util.Formatter;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.Mac;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import javax.security.auth.DestroyFailedException;

public class RdexCrypto {
    private static final int BLOCK_LENGTH = 16;
    private static final int BLOCK_LEN_SHA_256 = 32;
    private static final String ALGORITHM_SPEC = "AES/CBC/PKCS5PADDING";
    private enum KeyType { HMAC_KEY, CRYPTO_KEY }

    private final Rdex rdex;			//the app main window
    private final CardList mCardList;
    private boolean useDefaultPasswd = false;
    private char[] defaultPasswd = null;
    private String defaultPasswdHash = null;
    private SecretKey cryptoSessionKey = null;
    private SecretKey hmacSessionKey = null;
    private int cryptoSessionKeyHash = 0;
    private int hmacSessionKeyHash = 0;

    private File tempFile = null;       // temporary copy for decrypt because dialogs are not modal
    private Uri tempUri = null;         // temporary copy for decrypt because dialogs are not modal
    private long tempFileLen = 0;       // temporary copy for decrypt because dialogs are not modal
    private String tempErrorStr = "";   // temporary copy for decrypt because dialogs are not modal
    private boolean tempUpdateList = false;   // temporary copy for decrypt because dialogs are not modal

    public RdexCrypto(Rdex rdex, CardList cardList) {
        this.rdex = rdex;
        this.mCardList = cardList;
    }

    public boolean isUseDefaultPasswd() { return useDefaultPasswd; }

    //derive a session key from a hard coded password string
    //label allows us to generate multiple session keys from the same master password
    //examples from: https://stackoverflow.com/questions/992019/java-256-bit-aes-password-based-encryption
    private SecretKey getSessionKey(KeyType keyType, char[] passwd)
            throws NoSuchAlgorithmException, UnsupportedEncodingException {
        SecretKey sessionKey;
        String label = (keyType == KeyType.HMAC_KEY) ? "authentication" : "encryption";

        //algorithm used in MS WindowsCryptDeriveKey()
        MessageDigest digest = MessageDigest.getInstance("SHA-256");
        for (char c : passwd) {
            digest.update((Character.valueOf(c)).toString().getBytes("UTF-8"));
        }
        byte[] hash = digest.digest(label.getBytes("UTF-8"));

        if (keyType == KeyType.HMAC_KEY) {
            //use a 256-bit key for the hmac
            sessionKey = new SecretKeySpec(hash, "HmacSHA256");
        } else {
            //MS WindowsCryptDeriveKey() uses the first 128-bits for the crypto key
            sessionKey = new SecretKeySpec(hash, 0, BLOCK_LENGTH, "HmacSHA256");
        }
        mySecureZero(hash);
        return sessionKey;
    }

    private Mac initHmac(char[] passwd) {
        try {
            SecretKey sessionKey = hmacSessionKey;
            if (passwd != null)
                sessionKey = getSessionKey(KeyType.HMAC_KEY, passwd);
            Mac mac = Mac.getInstance("HmacSHA256");
            mac.init(sessionKey);
            return mac;
        }
        catch (NoSuchAlgorithmException e) {
            rdex.alertMsg("Init hmac NoSuchAlgorithmException");
            return null;
        } catch (InvalidKeyException e) {
            rdex.alertMsg("Init hmac InvalidKeyException");
            return null;
        } catch (UnsupportedEncodingException e) {
            rdex.alertMsg("Init hmac UnsupportedEncodingException");
            return null;
        }
    }

    //encrypt data, generate hmac tag and write to file
    public Rdex.Ternary encryptFile(String filename, Uri uri, String data) {
        File myFile = null;
        String errorStr;
        if (filename != null) {
            myFile = new File(filename);
            errorStr = "File \"" + rdex.extractFilePath(myFile.getAbsolutePath()) + "\" ";
        } else if (uri != null) {
            errorStr = "File from content server ";
        } else {
            rdex.alertMsg("Encrypt: No filename or content server specified.");
            return Rdex.Ternary.NEUTRAL;
        }

        char[] passwd = null;
        if (useDefaultPasswd) {
            if (defaultPasswd == null || defaultPasswd.length == 0) {
                rdex.alertMsg("Encrypt: no default passphrase set. " +
                        "You need to set the default passphrase before you can use it to encrypt a file.");
                return Rdex.Ternary.NEUTRAL;
            }
            if (!checkPasswdHash(defaultPasswd, defaultPasswdHash)) {
                rdex.alertMsg("Encrypt: " + errorStr + "default passphrase invalid. " +
                        "You will need to set the default passphrase again from the Encrypt menu");
                return Rdex.Ternary.NEUTRAL;
            }
            passwd = defaultPasswd;
        } else {
            if (cryptoSessionKey == null || hmacSessionKey == null) {
                rdex.alertMsg("Encrypt: " + errorStr + "no passphrase set.\n\nThere is a problem with the passphrase, " +
                        "you need to encrypt the file again to re-enter the passphrase.");
                return Rdex.Ternary.NEUTRAL;
            }
            if (cryptoSessionKey.hashCode() != cryptoSessionKeyHash || hmacSessionKey.hashCode() != hmacSessionKeyHash) {
                rdex.alertMsg("Encrypt: " + errorStr + "passphrase invalid.\n\nThere is a problem with the passphrase, " +
                        "you need to encrypt the file again to re-enter the passphrase.");
                return Rdex.Ternary.NEUTRAL;
            }
        }

        Mac hmac;
        byte[] iv, cyphertext;
        try {
            //generate a random initialization vector and initialize cipher with iv and session key
            SecretKey sessionKey = cryptoSessionKey;
            if (passwd != null)
                sessionKey = getSessionKey(KeyType.CRYPTO_KEY, passwd);
            Cipher cipher = Cipher.getInstance(ALGORITHM_SPEC);
            cipher.init(Cipher.ENCRYPT_MODE, sessionKey);
            AlgorithmParameters params = cipher.getParameters();
            iv = params.getParameterSpec(IvParameterSpec.class).getIV();

            //encode the data as UTF-8 and encrypt
            cyphertext = cipher.doFinal(data.getBytes("UTF-8"));
            if (iv == null || cyphertext == null) {
                //just in case
                rdex.alertMsg("Encrypt: " + errorStr + "write failure.");
                return Rdex.Ternary.NEUTRAL;
            }

            //initialize hmac and add iv and cyphertext
            hmac = initHmac(passwd);
            if (hmac == null) return Rdex.Ternary.NEUTRAL;
            hmac.update(iv);
            hmac.update(cyphertext);

        } catch (UnsupportedEncodingException e) {
            rdex.alertMsg("Encrypt: " + errorStr + "UnsupportedEncodingException"); return Rdex.Ternary.NEUTRAL;
        } catch (NoSuchAlgorithmException e) {
            rdex.alertMsg("Encrypt: " + errorStr + "NoSuchAlgorithmException"); return Rdex.Ternary.NEUTRAL;
        } catch (InvalidKeyException e) {
            rdex.alertMsg("Encrypt: " + errorStr + "InvalidKeyException"); return Rdex.Ternary.NEUTRAL;
        } catch (NoSuchPaddingException e) {
            rdex.alertMsg("Encrypt: " + errorStr + "NoSuchPaddingException"); return Rdex.Ternary.NEUTRAL;
        } catch (InvalidParameterSpecException e) {
            rdex.alertMsg("Encrypt: " + errorStr + "InvalidParameterSpecException"); return Rdex.Ternary.NEUTRAL;
        } catch (IllegalBlockSizeException e) {
            rdex.alertMsg("Encrypt: " + errorStr + "IllegalBlockSizeException"); return Rdex.Ternary.NEUTRAL;
        } catch (BadPaddingException e) {
            rdex.alertMsg("Encrypt: " + errorStr + "BadPaddingException"); return Rdex.Ternary.NEUTRAL;
        }

        // finally we commit ourselves and write the file
        OutputStream outStream = null;
        try {
            // need to explicitly truncate file from content server for api 29 by opening with mode 'wt'
            if (filename != null) outStream = new FileOutputStream(myFile);
            else outStream = rdex.getContentResolver().openOutputStream(uri, "wt");
            if (outStream == null) {
                rdex.alertMsg("Encrypt: " + errorStr + " failed to open for writing");
                return Rdex.Ternary.NEUTRAL;
            }
            //write rdex aes-128 format header
            String header = Rdex.RDEX_FILE_AES128_FORMAT_HDR;
            byte[] headerBytes = header.getBytes();
            outStream.write(headerBytes);
            outStream.write(CardList.SEPARATOR);

            //write iv, cyphertext and hmac tag to file
            outStream.write(iv);
            outStream.write(cyphertext);
            outStream.write(hmac.doFinal());
            return Rdex.Ternary.TRUE;
        } catch (FileNotFoundException e) {
            rdex.alertMsg("Encrypt: " + errorStr + " file not found exception"); return Rdex.Ternary.FALSE;
        } catch (IOException e) {
            rdex.alertMsg("Encrypt: " + errorStr + " write exception"); return Rdex.Ternary.FALSE;
        } finally {
            try {
                if (outStream != null) outStream.close();
            } catch (IOException e) {
                //quietly ignore IOException on closing
            }
        }
    }

    //check hmac agrees with hmac tag in file
    //return 1 if check passes, -1 if check fails without errors, 0 on error
    private int checkHmacTag(InputStream inStream, long fileLen, char[] passwd, String errorStr) {
        final int BUFSIZE = 4096;
        byte[] byteBuf = new byte[BUFSIZE];

        if (inStream == null) return 0;
        try {
            //skip rdex header
            long skipHdrLen = Rdex.RDEX_FILE_AES128_FORMAT_HDR.length() + 1;
            long bytesSkipped = inStream.skip(skipHdrLen);
            if (bytesSkipped != skipHdrLen) {
                rdex.alertMsg("Hmac check: " + errorStr + "skip rdex header failed");
                return 0;
            }

            //read iv and cyphertext into hmac
            Mac hmac = initHmac(passwd);
            if (hmac == null) return 0;
            int bytesRead, bytesToRead;
            long bytesRemaining = fileLen - skipHdrLen - BLOCK_LEN_SHA_256;
            if (bytesRemaining <= 0) {
                rdex.alertMsg("Hmac check: " + errorStr +
                        "length " + fileLen + " bytes is invalid");
                return 0;
            }
            bytesToRead = (bytesRemaining < BUFSIZE) ? (int) bytesRemaining : BUFSIZE;
            while (bytesRemaining > 0) {
                bytesRead = inStream.read(byteBuf, 0, bytesToRead);
                if (bytesRead != bytesToRead) {
                    rdex.alertMsg("Hmac check: " + errorStr + "error while reading file");
                    return 0;
                }
                hmac.update(byteBuf, 0, bytesRead);
                bytesRemaining -= bytesRead;
                bytesToRead = (bytesRemaining < BUFSIZE) ? (int) bytesRemaining : BUFSIZE;
            }
            byte[] hmacTag1 = hmac.doFinal();

            //read hmac tag from file
            byte[] hmacTag2 = new byte[BLOCK_LEN_SHA_256];
            bytesRead = inStream.read(hmacTag2);
            if (bytesRead != BLOCK_LEN_SHA_256) {
                rdex.alertMsg("Hmac check: " + errorStr + "eof error, bytes read " + bytesRead);
                return 0;
            }

            //check hmac tags for equality in a timing attack proof way
            int hmacCheck = 0;
            for (int i=0; i < BLOCK_LEN_SHA_256; ++i) {
                hmacCheck |= hmacTag1[i] ^ hmacTag2[i];
            }
            return (hmacCheck == 0) ? 1 : -1;
        }
        catch (IOException e) {
            rdex.alertMsg("Hmac check: " + errorStr + "file read error");
            return 0;
        }
        finally {
            try {
                inStream.close();
            } catch (IOException e) {
                rdex.alertMsg("Hmac check: " + errorStr + "file close error");
            }
        }
    }

    //skip rdex header, extract initialization vector from file and decrypt
    private boolean decryptValidatedCardlist(File myFile, Uri uri, long fileLen, char[] passwd, String errorStr) {
        final int BUFSIZE = 4096;
        byte[] byteBuf = new byte[BUFSIZE];
        byte[] iv = new byte[BLOCK_LENGTH];
        InputStream inStream = null;

        try {
            if (myFile != null) inStream = new FileInputStream(myFile);
            else inStream = rdex.getContentResolver().openInputStream(uri);
            if (inStream == null) {
                rdex.alertMsg("Decrypt: " + errorStr + "file access failed");
                return false;
            }

            //skip rdex header
            long skipHdrLen = Rdex.RDEX_FILE_AES128_FORMAT_HDR.length() + 1;
            long bytesSkipped = inStream.skip(skipHdrLen);
            if (bytesSkipped != skipHdrLen) {
                rdex.alertMsg("Decrypt: " + errorStr + "skip rdex header failed");
                return false;
            }

            //extract initialization vector
            int bytesRead, bytesToRead;
            bytesRead = inStream.read(iv);
            if (bytesRead != BLOCK_LENGTH) {
                rdex.alertMsg("Decrypt: " + errorStr + "failed to read initialization vector");
                return false;
            }

            //initialize cipher with iv and session key
            SecretKey sessionKey = cryptoSessionKey;
            if (passwd != null)
                sessionKey = getSessionKey(KeyType.CRYPTO_KEY, passwd);
            Cipher cipher = Cipher.getInstance(ALGORITHM_SPEC);
            cipher.init(Cipher.DECRYPT_MODE, sessionKey, new IvParameterSpec(iv));

            //read file to temp buffer ignoring final hmac tag
            long bytesRemaining = fileLen - skipHdrLen - BLOCK_LENGTH - BLOCK_LEN_SHA_256;
            if (bytesRemaining <= 0) {
                rdex.alertMsg("Decrypt: " + errorStr + "length " + fileLen + " bytes is invalid");
                return false;
            }
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            bytesToRead = (bytesRemaining < BUFSIZE) ? (int) bytesRemaining : BUFSIZE;
            while (bytesRemaining > 0) {
                bytesRead = inStream.read(byteBuf, 0, bytesToRead);
                if (bytesRead != bytesToRead) {
                    rdex.alertMsg("Decrypt: " + errorStr + "error while reading file");
                    return false;
                }
                bos.write(byteBuf, 0, bytesRead);
                bytesRemaining -= bytesRead;
                bytesToRead = (bytesRemaining < BUFSIZE) ? (int) bytesRemaining : BUFSIZE;
            }

            //decrypt and decode as UTF-8
            String cardStr = new String(cipher.doFinal(bos.toByteArray()), "UTF-8");
            mCardList.stringToCardList(cardStr);
            return true;
        }
        catch (IOException e) {
            rdex.alertMsg("Decrypt: " + errorStr + "file read error");
            return false;
        } catch (NoSuchAlgorithmException e) {
            rdex.alertMsg("Decrypt: " + errorStr + "NoSuchAlgorithmException");
            return false;
        } catch (InvalidKeyException e) {
            rdex.alertMsg("Decrypt: " + errorStr + "InvalidKeyException");
            return false;
        } catch (NoSuchPaddingException e) {
            rdex.alertMsg("Decrypt: " + errorStr + "NoSuchPaddingException");
            return false;
        } catch (InvalidAlgorithmParameterException e) {
            rdex.alertMsg("Decrypt: " + errorStr + "InvalidAlgorithmParameterException");
            return false;
        } catch (IllegalBlockSizeException e) {
            rdex.alertMsg("Decrypt: " + errorStr + "IllegalBlockSizeException");
            return false;
        } catch (BadPaddingException e) {
            rdex.alertMsg("Decrypt: " + errorStr + "BadPaddingException");
            return false;
        }
        finally {
            try {
                if (inStream != null) inStream.close();
            } catch (IOException e) {
                //quietly ignore an IOException on close
            }
        }
    }

    // public interface to decrypt file
    public Rdex.Ternary decryptCardlist(File myFile, Uri uri, String errorStr,
                                        boolean noDialog, boolean updateList) {
        long fileLen = -1;
        InputStream inStream;
        Cursor cursor = null;
        try {
            // open InputStream and get file length
            if (myFile != null) inStream = new FileInputStream(myFile);
            else inStream = rdex.getContentResolver().openInputStream(uri);
            if (myFile != null) {
                fileLen = myFile.length();
            } else {
                // get file length from content uri
                cursor = rdex.getContentResolver().query(uri, null, null, null, null);
                if (cursor != null && cursor.moveToFirst()) {
                    int sizeIndex = cursor.getColumnIndex(OpenableColumns.SIZE);
                    if (!cursor.isNull(sizeIndex)) {
                        fileLen = cursor.getLong(sizeIndex);
                    }
                }
                if (fileLen < 0) {
                    rdex.alertMsg("Decrypt: " + errorStr + " unknown file length");
                    return Rdex.Ternary.FALSE;
                }
            }
        } catch (FileNotFoundException e) {
            rdex.alertMsg("Decrypt: " + errorStr + " file not found");
            return Rdex.Ternary.FALSE;
        } catch (NullPointerException e) {
            rdex.alertMsg("Decrypt: " + errorStr + " null pointer exception");
            return Rdex.Ternary.FALSE;
        } finally {
            if (cursor != null) cursor.close();
        }

        if (checkPasswdHash(defaultPasswd, defaultPasswdHash)) {
            // the default password is valid, check it against the hmac tag
            int hmacCheck = checkHmacTag(inStream, fileLen, defaultPasswd, errorStr);
            if (hmacCheck == 1) {
                // hmac check succeeded, decrypt file with default passwd
                if (decryptValidatedCardlist(myFile, uri, fileLen, defaultPasswd, errorStr)) {
                    deleteCurrentPasswd();
                    useDefaultPasswd = true;
                    return Rdex.Ternary.TRUE;
                }
                else return Rdex.Ternary.FALSE;
            }
            else if (hmacCheck == 0) return Rdex.Ternary.FALSE;
        }

        if (inStream != null) {
            try { inStream.close(); }
            catch (IOException e) {
                rdex.alertMsg("Decrypt: " + errorStr + "I/O exception");
                return Rdex.Ternary.FALSE;
            }
        }

        // create temporary copies of parameters because Android dialogs are not modal
        tempFile = myFile;
        tempUri = uri;
        tempFileLen = fileLen;
        tempErrorStr = errorStr;
        tempUpdateList = updateList;

        if (noDialog) {
            // we are reopening the file after a save/restore instance state event, check the keys
            if (cryptoSessionKey == null || cryptoSessionKey.hashCode() != cryptoSessionKeyHash
                    || hmacSessionKey == null || hmacSessionKey.hashCode() != hmacSessionKeyHash) {
                // failed to restore the keys, ask user for passwd
                noDialog = false;
            }
        }

        if (noDialog) {
            // we are reopening the file after a save/restore instance state event
            // typically when the screen is rotated, finish opening the file
            Rdex.Ternary result = decryptDialogResponse(true, false, null);
            return (result == Rdex.Ternary.TRUE) ? Rdex.Ternary.TRUE : Rdex.Ternary.FALSE;
        } else {
            // ask the user for the password
            rdex.showDialogFragment(Rdex.DIALOG_DECRYPT_PASSWD_ID);
        }
        // file operation pending while waiting for user input
        return Rdex.Ternary.NEUTRAL;
    }

    // Android dialogs are not modal, response from password dialog -- finish decryption of the file
    // returns true for success, neutral for failed but state unchanged, false if failed and state modified
    public Rdex.Ternary decryptDialogResponse(boolean response, boolean fromDialog, Editable passwd) {
        boolean success = false;
        boolean cleanUp = false;
        // need to restore current keys in case of failure
        SecretKey tmpCryptoSessionKey = cryptoSessionKey;
        SecretKey tmpHmacSessionKey = hmacSessionKey;
        int tmpCryptoSessionKeyHash = cryptoSessionKeyHash;
        int tmpHmacSessionKeyHash = hmacSessionKeyHash;
        if (response && fromDialog) {
            // new password was entered in dialog (else restored by restoreInstanceState)
            response = setCryptoPasswdResult(passwd);
        }
        if (response) {
            // check password against the hmac tag
            int hmacCheck = 0;
            InputStream inStream = null;
            try {
                if (tempFile != null) inStream = new FileInputStream(tempFile);
                else inStream = rdex.getContentResolver().openInputStream(tempUri);
                if (inStream != null) {
                    hmacCheck = checkHmacTag(inStream, tempFileLen, null, tempErrorStr);
                }
            } catch (FileNotFoundException e) {
                rdex.alertMsg("Decrypt: " + tempErrorStr + " file not found");
            }
            finally {
                try {
                    if (inStream != null) inStream.close();
                }
                catch (IOException e) {
                    rdex.alertMsg("Decrypt: " + tempErrorStr + "I/O exception on close");
                }
            }

            if (hmacCheck == 1) {
                // hmac check succeeded, decrypt file with current passwd
                success = decryptValidatedCardlist(tempFile, tempUri, tempFileLen, null, tempErrorStr);
                if (success) useDefaultPasswd = false;
                // failure at this stage requires clean up
                cleanUp = !success;
            }
            else if (hmacCheck < 0) {
                // hmac check failed
                if (fromDialog) {
                    rdex.alertMsg("Decrypt: passphrase incorrect or file corrupted. " +
                            "The passphrase you entered will not decrypt this file.");
                } else {
                    rdex.alertMsg("Decrypt: passphrase incorrect. " +
                            "You will need to reenter the passphrase by opening the file again.");
                }
            }
        }
        if (success && fromDialog) {
            // store filename or uri and if file, update recent file list
            // (if noDialog filename and uri were restored from instance state)
            // updateList prevents us from adding a content uri from an incoming intent at startup
            // to the recent files list
            rdex.updateFileLoadState(tempFile, tempUri, tempUpdateList);
        }
        if (response && fromDialog && !success) {
            // restore previous keys
            cryptoSessionKey = tmpCryptoSessionKey;
            hmacSessionKey = tmpHmacSessionKey;
            cryptoSessionKeyHash = tmpCryptoSessionKeyHash;
            hmacSessionKeyHash = tmpHmacSessionKeyHash;
        }
        tempFile = null;
        tempUri = null;
        tempFileLen = 0;
        tempErrorStr = "";
        tempUpdateList = false;
        if (cleanUp) return Rdex.Ternary.FALSE;
        return success ? Rdex.Ternary.TRUE : Rdex.Ternary.NEUTRAL;
    }

    private String createPasswdHash(char[] passwd) throws NoSuchAlgorithmException, UnsupportedEncodingException {
        MessageDigest digest = MessageDigest.getInstance("SHA-256");
        for (char c : passwd) {
            digest.update((Character.valueOf(c)).toString().getBytes("UTF-8"));
        }
        byte[] hash = digest.digest();
        Formatter formatter = new Formatter();
        for (byte b : hash) {
            // format into hex, easier to handle than base64
            formatter.format("%02x", b);
        }
        return formatter.toString();
    }

    // check for bit rot in the password
    private boolean checkPasswdHash(char[] passwd, String passwdHash) {
        if (passwd == null || passwd.length == 0) return false;
        try {
            return passwdHash.equals(createPasswdHash(passwd));
        }
        catch (NoSuchAlgorithmException e) {
            rdex.alertMsg("Check passphrasae hash: NoSuchAlgorithmException");
        }
        catch (UnsupportedEncodingException e) {
            rdex.alertMsg("Check passphrasae hash: UnsupportedEncodingException");
        }
        return false;
    }

    // result from set default password dialog
    public void setDefaultPasswdResult(Editable passwd) {
        SharedPreferences settings = rdex.getPreferences(0);
        SharedPreferences.Editor prefsEditor = settings.edit();
        if (passwd == null || passwd.length() == 0) {
            // clear default passwd
            if (useDefaultPasswd) rdex.alertMsg(
                "You have a file open that uses the default passphrase. " +
                "To save this file you will either need to set a new default passphrase " +
                "or use the \"Encrypt\" menu to select a unique passphrase for the file.");
            rdex.showStatusMsg("Default passphrase cleared.");
        } else {
            int len = passwd.length();
            defaultPasswd = new char[len];
            passwd.getChars(0, len, defaultPasswd, 0);
            passwd.clear();
            try {
                defaultPasswdHash = createPasswdHash(defaultPasswd);
                prefsEditor.putString(Rdex.KEY_SETTINGS_DEFAULT_PASSWD, new String(defaultPasswd));
                prefsEditor.putString(Rdex.KEY_SETTINGS_DEFAULT_PASSWD_HASH, defaultPasswdHash);
                prefsEditor.apply();
                if (useDefaultPasswd) {
                    new AlertDialog.Builder(rdex)
                        .setTitle(R.string.change_passwd_dialog_title)
                        .setMessage(R.string.change_passwd_dialog)
                        .setPositiveButton("Yes", new DialogInterface.OnClickListener() {
                            @Override
                            public void onClick(DialogInterface dialog, int id) {
                                rdex.saveFile();
                            }
                        })
                        .setNegativeButton("No", new DialogInterface.OnClickListener() {
                            @Override
                            public void onClick(DialogInterface dialog, int id) { }
                        })
                        .show();
                }
                else rdex.showStatusMsg("Default passphrase set.");
                return;
            } catch (NoSuchAlgorithmException e) {
                rdex.alertMsg("Set default passphrase: NoSuchAlgorithmException");
            } catch (UnsupportedEncodingException e) {
                rdex.alertMsg("Check passphrasae hash: UnsupportedEncodingException");
            }
        }
        mySecureZero(defaultPasswd);
        defaultPasswd = null;
        defaultPasswdHash = null;
        prefsEditor.putString(Rdex.KEY_SETTINGS_DEFAULT_PASSWD, "");
        prefsEditor.putString(Rdex.KEY_SETTINGS_DEFAULT_PASSWD_HASH, "");
        prefsEditor.apply();
    }

    // result from set encryption or decryption password dialog
    public boolean setCryptoPasswdResult(Editable passwd) {
        if (passwd == null || passwd.length() == 0) {
            rdex.alertMsg("Set encrypt/decrypt passphrase: passphrase field is empty");
            return false;
        }
        deleteCurrentPasswd();
        int len = passwd.length();
        char[] newPasswd = new char[len];
        passwd.getChars(0, len, newPasswd, 0);
        passwd.clear();
        try {
            cryptoSessionKey = getSessionKey(KeyType.CRYPTO_KEY, newPasswd);
            hmacSessionKey = getSessionKey(KeyType.HMAC_KEY, newPasswd);
            cryptoSessionKeyHash = cryptoSessionKey.hashCode();
            hmacSessionKeyHash = hmacSessionKey.hashCode();
            mySecureZero(newPasswd);
            return true;
        }
        catch (NoSuchAlgorithmException e) {
            rdex.alertMsg("Set encrypt/decrypt passphrase: NoSuchAlgorithmException");
            deleteCurrentPasswd();
            mySecureZero(newPasswd);
            return false;
        } catch (UnsupportedEncodingException e) {
            rdex.alertMsg("Set encrypt/decrypt passphrase: UnsupportedEncodingException");
            deleteCurrentPasswd();
            mySecureZero(newPasswd);
            return false;
        }
    }

    // result from use default password dialog
    public void setUseDefaultPasswd() {
        deleteCurrentPasswd();
        useDefaultPasswd = true;
    }

    public void resetUseDefaultPasswd() {
        useDefaultPasswd = false;
    }

    // set default password from preferences
    public void setDefaultPasswdPref(String passwd, String passwdHash) {
        if (passwd == null || passwd.length() == 0) {
            defaultPasswd = null;
            defaultPasswdHash = null;
        } else {
            defaultPasswd = passwd.toCharArray();
            defaultPasswdHash = passwdHash;
        }
    }

    public void saveCryptoInstanceState(Bundle outState) {
        // save instance state when the device is rotated or memory reclaimed by the OS
        // it also appears to be called when we start another activity such as edit card
        // so we can't delete the current password
        byte[] key = (cryptoSessionKey != null) ? cryptoSessionKey.getEncoded() : null;
        byte[] hmac = (hmacSessionKey != null) ? hmacSessionKey.getEncoded() : null;
        outState.putByteArray(Rdex.KEY_CRYPTO_SESSION_KEY, key);
        outState.putInt(Rdex.KEY_CRYPTO_SESSION_KEY_HASH, cryptoSessionKeyHash);
        outState.putByteArray(Rdex.KEY_HMAC_SESSION_KEY, hmac);
        outState.putInt(Rdex.KEY_HMAC_SESSION_KEY_HASH, hmacSessionKeyHash);
    }

    public void restoreCryptoInstanceState(Bundle savedInstanceState) {
        byte[] key = savedInstanceState.getByteArray(Rdex.KEY_CRYPTO_SESSION_KEY);
        cryptoSessionKeyHash = savedInstanceState.getInt(Rdex.KEY_CRYPTO_SESSION_KEY_HASH);
        byte[] hmac = savedInstanceState.getByteArray(Rdex.KEY_HMAC_SESSION_KEY);
        hmacSessionKeyHash = savedInstanceState.getInt(Rdex.KEY_HMAC_SESSION_KEY_HASH);
        if (key != null && hmac != null) {
            cryptoSessionKey = new SecretKeySpec(key, "HmacSHA256");
            hmacSessionKey = new SecretKeySpec(hmac, "HmacSHA256");
        }
        mySecureZero(key);
        mySecureZero(hmac);
    }

    public boolean isDefaultPasswdSet() {
        return defaultPasswd != null && defaultPasswd.length != 0;
    }

    public void deleteCurrentPasswd() {
        if (Build.VERSION.SDK_INT >= 26) {
            try {
                //destroy only has a default implementation on api 26 and later
                if (cryptoSessionKey != null) cryptoSessionKey.destroy();
                if (hmacSessionKey != null) hmacSessionKey.destroy();
            }
            catch (DestroyFailedException ignored) { }
        }
        cryptoSessionKey = null;
        hmacSessionKey = null;
        cryptoSessionKeyHash = 0;
        hmacSessionKeyHash = 0;
    }

    private void mySecureZero(char[] passwd) {
        if (passwd == null || passwd.length == 0) return;
        Arrays.fill(passwd, (char) 0);
    }

    private void mySecureZero(byte[] passwd) {
        if (passwd == null || passwd.length == 0) return;
        Arrays.fill(passwd, (byte) 0);
    }
}
