/*
Rdex version 2.4 for Android

Copyright (C) 2018 Peter Newman <pn@pnewman.com>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE.  See the GNU General Public License for more details:
<http://www.gnu.org/licenses/>.
*/

package com.pnewman.rdex;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;


public class CardList {
	
	private final Rdex rdex;	//the app main window
    private int numCards;
    private int cardIndex;		//text search ptr within current card
    private Card cardListHead;
    private Card currentCard;

    static final char SEPARATOR = '\177';

    private int endIndex;	//cheap way to return end index after wildcard search
    private String patternLower;	//used for wildcard search
    private String patternUpper;	//used for wildcard search

    private class Card {
        Card nextCard;
        Card prevCard;
        int cardId;
        String text;
        
        public Card(String str) {
            nextCard = null;
            prevCard = null;
            cardId = 0;
            text = str;
        }

        public void setNextCard(Card next) { nextCard = next; }
        public Card getNextCard() { return nextCard; }
        public void setPrevCard(Card prev) { prevCard = prev; }
        public Card getPrevCard() { return prevCard; }
        public void setCardId(int id) { cardId = id; }
        public int getCardId() { return cardId; }
        public void setCardText(String str) { text = str; }
        public String getCardText() { return text; }

        // simple wildcard search
        // from Rob Pike's regular expression matcher
        // http://www.cs.princeton.edu/courses/archive/spr10/cos333/beautiful.html
        public int searchCardWild(int patternIndex, int index) {
        	if (patternIndex < patternLower.length()) {
	        	do {
	        		endIndex = matchChar(patternIndex, index);
	        		if (endIndex > -1) {
	        			return index;
	        		}
	        	} while (index++ < text.length());
        	}
        	return -1;
        }
        
        private int matchChar(int patternIndex, int textIndex) {
        	if (patternIndex >= patternLower.length()) {
        		return textIndex;
        	}
        	if (patternLower.charAt(patternIndex) == '*') {
        		return matchStar(patternIndex + 1, textIndex);
        	}
			if (textIndex >= text.length()) {
				return -1;
			}
		    if (patternLower.charAt(patternIndex) == '\\') {
		    	//following char should be interpreted literally
		        return matchLiteral(patternIndex + 1, textIndex);
		    }
		    if (text.charAt(textIndex) == patternLower.charAt(patternIndex) ||
		    		text.charAt(textIndex) == patternUpper.charAt(patternIndex) ||
		    		patternLower.charAt(patternIndex) == '?') {
				// match the next character in the pattern
		    	return matchChar(patternIndex + 1, textIndex + 1);
			}
			return -1;
        }

        private int matchLiteral(int patternIndex, int textIndex) {
        	if (patternIndex >= patternLower.length()) {
        		return textIndex;
        	}
			if (text.charAt(textIndex) == patternLower.charAt(patternIndex) ||
			    		text.charAt(textIndex) == patternUpper.charAt(patternIndex)) {
				// match the next character in the pattern
		    	return matchChar(patternIndex + 1, textIndex + 1);
			}
			return -1;
        }
        
        private int matchStar(int patternIndex, int textIndex) {
        	int limit = Rdex.MAX_WILDCARD_SEARCH;
        	while (patternIndex < patternLower.length() &&
        			patternLower.charAt(patternIndex) == '*') {
        		patternIndex++;
        		limit += Rdex.MAX_WILDCARD_SEARCH;
        	}
        	if (patternIndex >= patternLower.length()) {
        		return textIndex;
        	}
            do {
            	endIndex = matchChar(patternIndex, textIndex);
                if (endIndex > -1) return endIndex;
            } while (textIndex++ < text.length() && --limit > 0);
            return -1;
        }
    }

    public CardList(Rdex rdex) {
    	this.rdex = rdex;
    	resetCardList();
    }

    
    public int getNumCards() { return numCards; }
    public int getCardIndex() { return cardIndex; }
    public void setCardIndex(int index) { cardIndex = index; }
    

    public String getCardText() {
    	if (numCards == 0 || currentCard == null) return null;
    	return currentCard.getCardText();
    }
    
    
    public void addNewCard(String text) {
    	addCard(text);
        currentCard = cardListHead.getPrevCard();
    }
    
    
    // append card to tail of circular card list
    public void addCard(String text) {
        Card newCard = new Card(text);
        if (numCards > 0) {
            newCard.setNextCard(cardListHead);
            newCard.setPrevCard(cardListHead.getPrevCard());
            newCard.setCardId(cardListHead.getPrevCard().getCardId() + 1);
            cardListHead.getPrevCard().setNextCard(newCard);
            cardListHead.setPrevCard(newCard);
        } else {
            //add first card
            newCard.setNextCard(newCard);
            newCard.setPrevCard(newCard);
            newCard.setCardId(1);
            currentCard = cardListHead = newCard;
        }
        numCards++;
    }

    
    public boolean deleteCard() {
    	if (numCards < 1 || currentCard == null) {
    		currentCard = null;
        	showCard();
    		return false;
    	}

    	if (numCards == 1) {
    		resetCardList();
    	} else {
    		Card cardPtr = currentCard;
    		cardPtr.getPrevCard().setNextCard(cardPtr.getNextCard());
    		cardPtr.getNextCard().setPrevCard(cardPtr.getPrevCard());
    		currentCard = cardPtr.getNextCard();
    		int cardId = cardPtr.getCardId();
    		boolean lastCardOnList = (cardPtr.getNextCard() == cardListHead);
    		if (cardPtr == cardListHead) {
    			//deleting card at head of list
    			cardListHead = currentCard;
    			cardId = 1;
    		}
    		--numCards;
    		cardIndex = 0;
    		if (!lastCardOnList) renumberCards(cardId);
    	}
    	showCard();
    	return true;
    }
    

    //renumber cards starting at current card with given card id
    private void renumberCards(int cardId) {
    	int i = 0;
    	Card cardPtr = currentCard;
    	while ((cardPtr != cardListHead || i == 0) && i < numCards) {
    		cardPtr.setCardId(cardId++);
    		cardPtr = cardPtr.getNextCard();
    		++i;
    	}
    }


    public int getCurrentCardId() {
    	if (currentCard == null) return -1;
    	return currentCard.getCardId();
    }


    //search list to find card given cardId
    public void setCurrentCardId(int cardId) {
    	int i = 0;
    	Card cardPtr = cardListHead;
    	while (cardPtr.getCardId() != cardId && i < numCards) {
    		cardPtr = cardPtr.getNextCard();
    		++i;
    	}
    	if (cardPtr.getCardId() == cardId) currentCard = cardPtr;
    	else currentCard = cardListHead;
    }

    
    //discard all previous cards
    public void resetCardList() {
    	numCards = cardIndex = 0;
    	currentCard = cardListHead = null;
    }

    
    //replace text of current card
    public void updateCurrentCard(String str) {
    	if (numCards < 1) return;
        currentCard.setCardText(str);
    }

    
    public void nextCard() {
    	if (numCards < 1) return;
    	currentCard = currentCard.getNextCard();
    	showCard();
    }

    
    public void previousCard() {
    	if (numCards < 1) return;
    	currentCard = currentCard.getPrevCard();
    	showCard();
    }

    
    private void showStatusMsg(String msg) {
    	rdex.showStatusMsg(msg);
    }

    
    public void showCard() {
    	if (currentCard != null)
    		rdex.showCard(currentCard.getCardText());
    	else
    		rdex.showCard("");
    }

    
    //show card and mark selected text
    public void showCard(int begin, int end) {
    	if (currentCard != null) {
    		rdex.showCard(currentCard.getCardText(), begin, end);
    	} else {
    		rdex.showCard("");
    	}
    }


    //copy pattern to class variables and remove leading '*'s
    private int setupPattern(String pattern) {
    	if (pattern.length() == 0) return 0;
		patternLower = pattern.toLowerCase();
		patternUpper = pattern.toUpperCase();
		int patternIndex = 0;
		for (int i=0; i < pattern.length(); ++i) {
			//remove leading '*'s from pattern
			if (pattern.charAt(i) == '*') patternIndex = i + 1;
			else if (pattern.charAt(i) != '?') break;
		}
		return patternIndex;
    }
    
    
    public void searchFwd(String pattern) {
    	if (numCards < 1) return;
        if (pattern.length() == 0) {
            nextCard();
        } else {
            Card cardPtr = currentCard;
            Card endPtr = currentCard;
            int patternIndex = setupPattern(pattern);
            int index = cardPtr.searchCardWild(patternIndex, cardIndex + 1);
            if (index < 0) {
                // string not found in remainder of current card
                do {
                    // search all cards
                    cardPtr = cardPtr.getNextCard();
                    index = cardPtr.searchCardWild(patternIndex, 0);
                } while (cardPtr != endPtr && index < 0);
            }
            if (index > -1) {
                // found string
                currentCard = cardPtr;
                cardIndex = index;
                showCard(index, endIndex);
            } else {
            	showStatusMsg("\"" + pattern + "\" not found");
            }
        }
    }

    
    //stores a card title line and index so that a list of them can be sorted
    public static class CardTitle implements Comparable<CardTitle> {
        private int cardIndex;
        private String cardTitle;

        @Override
        public int compareTo(CardTitle otherTitle) {
            return cardTitle.compareTo(otherTitle.cardTitle);
        }
    }


    // create list of card titles for all cards that contain search string
	public boolean listCards(String pattern,
			ArrayList<Integer> cardIdList, ArrayList<String> titleList, boolean listTitlesAlpha)
	{	
		if (numCards < 1) return true;
		ArrayList<CardTitle> cardTitleList = new ArrayList<>();
		CardTitle titleEntry;
        String cardText;
	    int index;
        Card cardPtr = cardListHead.getPrevCard();
        int patternIndex = setupPattern(pattern);
	    for (int i=0; i < numCards; i++) {
            cardPtr = cardPtr.getNextCard();
	        if (pattern.length() > 0) {
	            // search for string in card
                index = cardPtr.searchCardWild(patternIndex, 0);
	            // skip card if pattern not found
	            if (index < 0) continue;
	        }   
	        // remove leading whitespace and extract first line of card
            cardText = cardPtr.getCardText().trim();
	        index = cardText.indexOf('\n');
	        if (index < 0) {
                // card has no newline characters
                index = cardText.length();
                if (index < 1)  {
                    index = 12;
                    cardText = "<Empty Card>";
                }
            }
	        if (index > 40) index = 40;
	        // add a reference to the list
	        titleEntry = new CardTitle();
	        titleEntry.cardIndex = cardPtr.getCardId();
	        titleEntry.cardTitle = cardText.substring(0, index);
	        cardTitleList.add(titleEntry);
            if (cardTitleList.size() >= Rdex.MAX_LIST_ALL_ITEMS) {
                // too many items in list
                return false;
            }
	    }
	    
        if (listTitlesAlpha) Collections.sort(cardTitleList, new Comparator<CardTitle>() {
            @Override
            public int compare(CardTitle card1, CardTitle card2) {
                return card1.cardTitle.compareToIgnoreCase(card2.cardTitle);
            }
        });
	    for (CardTitle c : cardTitleList) {
	    	titleList.add(c.cardTitle);      
	        cardIdList.add(c.cardIndex);
	    }
        return true;
	}

    // convert string from load file to list of cards
    public void stringToCardList(String buffer) {
        int startIndex = 0;
        int endIndex;
        int bufferLen = buffer.length();
        if (bufferLen == 0) return;

        //discard all previous cards
        resetCardList();

        while (startIndex < bufferLen) {
            endIndex = buffer.indexOf(SEPARATOR, startIndex);
            if (endIndex >= 0) {
                addCard(buffer.substring(startIndex, endIndex));
                startIndex = endIndex + 1;
            } else {
                //discard any chars after final separator
                break;
            }
        }
        showCard();
    }

    // convert card list to string for file saving
    public String cardListToString(boolean noFinalSeparator) {
        Card cardPtr = cardListHead;
        StringBuilder buffer = new StringBuilder();
        if (numCards < 1) return null;
        if (numCards == 1 && noFinalSeparator) {
            //plain text file, do not add separator
            buffer.append(cardPtr.getCardText());
        } else {
            for (int i = 0; i < numCards; i++) {
                buffer.append(cardPtr.getCardText());
                buffer.append(SEPARATOR);
                cardPtr = cardPtr.getNextCard();
            }
        }
        return buffer.toString();
    }
}
