/*
Palm Keyring for Android

Copyright (C) 2022 Peter Newman <pn@pnewman.com>

Palm Keyring for Android is based on:
KeyringEditor v1.1
Copyright 2004 Markus Griessnig
Vienna University of Technology Institute of Computer Technology

KeyringEditor is based on:
Java Keyring v0.6
Copyright 2004 Frank Taylor <keyring@lieder.me.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE.  See the GNU General Public License for more details:
<http://www.gnu.org/licenses/>.
*/

// 27.08.2022: change default read only mode to disabled

package com.pnewman.keyring;

import android.app.Activity;
import android.content.SharedPreferences;
import android.os.Build;
import android.os.Bundle;
import android.view.View;
import android.view.View.OnClickListener;
import android.widget.AdapterView;
import android.widget.AdapterView.OnItemSelectedListener;
import android.widget.ArrayAdapter;
import android.widget.CheckBox;
import android.widget.Spinner;

public class Preferences extends Activity {

	public static final int SIZE_START_VALUE = 8;
	public static final int SIZE_ARRAY_MAX = 10;
    public static final int NUM_FONTS = 4;
    public static final int NUM_STYLES = 4;
    public static final int NUM_TIMEOUT_VALUES = 5;
    public static final int DEFAULT_TEXT_SIZE = 16;
    public static final boolean DEFAULT_READ_ONLY_MODE = false;


    private SharedPreferences settings;
    
	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
	    setContentView(R.layout.preferences);
        setTitle(R.string.preferences_dialog_title);

	    settings = getSharedPreferences("Keyring", 0);

		//initialize text font spinner
	    Spinner fontSpinner = findViewById(R.id.font_spinner);
	    ArrayAdapter<CharSequence> fontAdapter = ArrayAdapter.createFromResource(
	            this, R.array.font_array, android.R.layout.simple_spinner_item);
	    fontAdapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
		if (Build.VERSION.SDK_INT >= 23)
			// (fixes problem of dark text on dark background)
			fontAdapter.setDropDownViewTheme(this.getTheme());
	    fontSpinner.setAdapter(fontAdapter);
	    int fontSetting = settings.getInt(Keyring.KEY_SETTINGS_FONT, 0);
	    if (fontSetting < 0 || fontSetting >= NUM_FONTS) fontSetting = 0;
	    fontSpinner.setSelection(fontSetting);
	    
        //initialize text style spinner
	    Spinner styleSpinner = findViewById(R.id.style_spinner);
	    ArrayAdapter<CharSequence> styleAdapter = ArrayAdapter.createFromResource(
	            this, R.array.style_array, android.R.layout.simple_spinner_item);
	    styleAdapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
		if (Build.VERSION.SDK_INT >= 23)
			styleAdapter.setDropDownViewTheme(this.getTheme());
	    styleSpinner.setAdapter(styleAdapter);
	    int styleSetting = settings.getInt(Keyring.KEY_SETTINGS_STYLE, 0);
	    if (styleSetting < 0 || styleSetting >= NUM_STYLES) styleSetting = 0;
	    styleSpinner.setSelection(styleSetting);
	    
        //initialize text size spinner
	    String[] sizeArray = new String[SIZE_ARRAY_MAX];
	    for (int i=0; i < SIZE_ARRAY_MAX; ++i) {
	    	sizeArray[i] = Integer.toString(i * 2 + SIZE_START_VALUE);
	    }
	    Spinner sizeSpinner = findViewById(R.id.size_spinner);
	    ArrayAdapter<CharSequence> sizeAdapter = new ArrayAdapter<CharSequence>(this,
	    		android.R.layout.simple_spinner_item, sizeArray);
		if (Build.VERSION.SDK_INT >= 23)
			sizeAdapter.setDropDownViewTheme(this.getTheme());
	    sizeSpinner.setAdapter(sizeAdapter);
	    int sizeSetting = settings.getInt(Keyring.KEY_SETTINGS_SIZE, DEFAULT_TEXT_SIZE);
	    sizeSetting = (sizeSetting - SIZE_START_VALUE) / 2;
	    if (sizeSetting < 0 || sizeSetting >= SIZE_ARRAY_MAX)
	    	sizeSetting = (DEFAULT_TEXT_SIZE - SIZE_START_VALUE) / 2;
	    sizeSpinner.setSelection(sizeSetting);
	    
        //initialize lock timeout spinner
	    Spinner timeoutSpinner = findViewById(R.id.timeout_spinner);
	    ArrayAdapter<CharSequence> timeoutAdapter = ArrayAdapter.createFromResource(
	            this, R.array.timeout_array, android.R.layout.simple_spinner_item);
		timeoutAdapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
		if (Build.VERSION.SDK_INT >= 23)
			timeoutAdapter.setDropDownViewTheme(this.getTheme());
	    timeoutSpinner.setAdapter(timeoutAdapter);
	    int timeoutSetting = settings.getInt(Keyring.KEY_SETTINGS_TIMEOUT, 3);
	    if (timeoutSetting < 0 || timeoutSetting >= NUM_TIMEOUT_VALUES) timeoutSetting = 0;
	    timeoutSpinner.setSelection(timeoutSetting);
	    
	    //initialize text font callback
		fontSpinner.setOnItemSelectedListener(new OnItemSelectedListener() {
	    	@Override
		    public void onItemSelected(AdapterView<?> parent, View view, int pos, long id) {
	    	    int fontSetting = settings.getInt(Keyring.KEY_SETTINGS_FONT, 0);
	    	    if (fontSetting < 0 || fontSetting >= NUM_FONTS) fontSetting = 0;
	    	    if (pos != fontSetting) {
	    			SharedPreferences.Editor editor = settings.edit();
					editor.putInt(Keyring.KEY_SETTINGS_FONT, pos);
					editor.apply();
	    	    }
	    	}

		    public void onNothingSelected(AdapterView<?> parent) { }
	    });

        //initialize text style callback
		styleSpinner.setOnItemSelectedListener(new OnItemSelectedListener() {
	    	@Override
		    public void onItemSelected(AdapterView<?> parent, View view, int pos, long id) {
	    	    int styleSetting = settings.getInt(Keyring.KEY_SETTINGS_STYLE, 0);
	    	    if (styleSetting < 0 || styleSetting >= NUM_STYLES) styleSetting = 0;
	    	    if (pos != styleSetting) {
	    			SharedPreferences.Editor editor = settings.edit();
					editor.putInt(Keyring.KEY_SETTINGS_STYLE, pos);
					editor.apply();
	    	    }
	    	}

		    public void onNothingSelected(AdapterView<?> parent) { }
	    });	

        //initialize text size callback
		sizeSpinner.setOnItemSelectedListener(new OnItemSelectedListener() {
	    	@Override
		    public void onItemSelected(AdapterView<?> parent, View view, int pos, long id) {
	    	    int sizeSetting = settings.getInt(Keyring.KEY_SETTINGS_SIZE, DEFAULT_TEXT_SIZE);
	    	    sizeSetting = (sizeSetting - SIZE_START_VALUE) / 2;
	    	    if (sizeSetting < 0 || sizeSetting >= SIZE_ARRAY_MAX)
	    	    	sizeSetting = (DEFAULT_TEXT_SIZE - SIZE_START_VALUE) / 2;
	    	    if (pos != sizeSetting) {
	    			SharedPreferences.Editor editor = settings.edit();
					editor.putInt(Keyring.KEY_SETTINGS_SIZE, pos * 2 + SIZE_START_VALUE);
					editor.apply();
	    	    }
	    	}

		    public void onNothingSelected(AdapterView<?> parent) { }
	    });	

		//initialize timeout value callback
		timeoutSpinner.setOnItemSelectedListener(new OnItemSelectedListener() {
	    	@Override
		    public void onItemSelected(AdapterView<?> parent, View view, int pos, long id) {
	    	    int timeoutSetting = settings.getInt(Keyring.KEY_SETTINGS_TIMEOUT, 3);
	    	    if (timeoutSetting < 0 || timeoutSetting >= NUM_TIMEOUT_VALUES) timeoutSetting = 0;
	    	    if (pos != timeoutSetting) {
	    			SharedPreferences.Editor editor = settings.edit();
					editor.putInt(Keyring.KEY_SETTINGS_TIMEOUT, pos);
					editor.apply();
	    	    }
	    	}

		    public void onNothingSelected(AdapterView<?> parent) { }
	    });

	    //initialize read only checkbox and callback
	    final CheckBox checkbox = findViewById(R.id.checkbox);
	    boolean readOnly =
	    		settings.getBoolean(Keyring.KEY_SETTINGS_READ_ONLY_MODE, DEFAULT_READ_ONLY_MODE);
	    checkbox.setChecked(readOnly);
	    checkbox.setOnClickListener(new OnClickListener() {
	        public void onClick(View v) {
    			SharedPreferences.Editor editor = settings.edit();
    			editor.putBoolean(Keyring.KEY_SETTINGS_READ_ONLY_MODE, ((CheckBox) v).isChecked());
				editor.apply();
			}
			    });

	}
}
