/*
Palm Keyring for Android

Copyright (C) 2022 Peter Newman <pn@pnewman.com>

Palm Keyring for Android is based on:
KeyringEditor v1.1
Copyright 2004 Markus Griessnig
Vienna University of Technology
Institute of Computer Technology

KeyringEditor is based on:
Java Keyring v0.6
Copyright 2004 Frank Taylor <keyring@lieder.me.uk>

These programs are distributed in the hope that they will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.
*/

// Model.java

// 29.10.2004

// 31.10.2004: backup header and categories for saveData()
// 02.11.2004: class entry changed; added parameter -e; added parameter -d
// 03.11.2004: added getDateType()
// 04.11.2004: added getDataFormat()
// 06.11.2004: added debugByteArray()
// 08.11.2004: Categories-Array with 276 byte
// 11.11.2004: addes elements(), getCategoryName(), getCategories(); loadData - empty title possible
// 17.11.2004: setPassword uses char[] (security reason)
// 23.11.2004: added saveEntriesToFile()
// 24.11.2004: updated saveEntriesToFile(); updated saveData()
// 30.11.2004: printHexByteArray() added
// 01.12.2004: Keyring database format 5 support added
// 02.12.2004: toRecordFormat5() added
// 05.12.2004: convertDatabase() added
// 07.12.2004: convertDatabase() updated
// 12.01.2004: writeNewDatabase() added
// 07.09.2005: loadDatabase() ignores deleted record table entries
// 23.09.2005: added getNewUniqueId()
// 12.06.2021: change file load to remove limit on file length
// 04.09.2021: restore original file loading code for small files, problem reported by user
// 25.09.2021: add default Windows-1252 character encoding to load, display and store
// 25.09.2021: check record lengths before writing to file
// 08.01.2022: detect vCard file format with a nice failure message
// 20.01.2022: support zero length entries in file loading for original Palm Keyring files
// 20.08.2022: add init new database for new database file creation

package com.pnewman.keyring;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Enumeration;
import java.util.GregorianCalendar;
import java.util.Vector;

/**
 * This class is used to load and save Keyring databases.
 */
public class Model {
	// ----------------------------------------------------------------
	// variables
	// ----------------------------------------------------------------

	public static final boolean DEBUG = false;

	/**
	 * Max file length for reading with original file loading code.
	 */
	public static final int BUFFER_SIZE = 200 * 1024;

	/**
	 * Max single entry size for unlimited file length loading code.
	 */
	public static final int MAX_ENTRY_SIZE = 100 * 1024;

	// PDB header information (readPDBHeader)
	/**
	 * Header of Keyring database
	 */
	private byte[] pdbHeader = new byte[78];

	/**
	 * Categories in Keyring database
	 */
	private byte[] pdbCategories = new byte[276];

	private String pdbName;        // 32
	private int pdbFlags;          // 2 (unsigned)

	/**
	 * Keyring database version
	 */
	protected int pdbVersion;      // 2 (unsigned) // Keyring database format
	private long pdbModNumber;     // 4 (unsigned), modification number
	private int pdbSortInfoOffset; // 4
	private String pdbType;        // 4
	private String pdbCreator;     // 4
	private int pdbAppInfoOffset;  // 4

	/**
	 * Number of records in the keyring database
	 */
	private int pdbNumRecords;     // 2

	// Keyring database format 4
	private int recordZeroAttribute;
	private int recordZeroUniqueId;
	private int recordZeroLength;

	/**
	 * Number of iterations for keyring database version 5
	 */
	private int pdbIterations = 0;

	/**
	 * Vector to entry objects
	 */
	private final Vector<Entry> entries = new Vector<>(); // reference to entry objects

	/**
	 * Vector to category strings
	 */
	private Vector<String> categories = new Vector<>(); // category labels

	/**
	 * Reference to class Crypto
	 */
	protected Crypto crypto; // Gui.java

	/**
	 * Character encoding expected in database file format
	 */
	private static String usingCharset = "ISO-8859-1";

	/**
	 * Constructor sets encoding expected in database file format
	 */
	public Model() {
		if (Charset.isSupported("windows-1252"))
			usingCharset = "windows-1252";
		if(DEBUG)
			System.out.println("Using " + usingCharset + " character encoding");
	}

	// ----------------------------------------------------------------
	// public ---------------------------------------------------------
	// ----------------------------------------------------------------

	// entries --------------------------------------------------------
	/**
	 * This method adds an entry to the vector entries.
	 *
	 * @param entry Entry object
	 */
	public void addEntry(Object entry) {
		entries.add((Entry)entry);
	}

	/**
	 * This method removes an entry from the vector entries.
	 *
	 * @param entry Entry object
	 */
	public void removeEntry(Object entry) {
		entries.removeElement(entry);
	}

	/**
	 * This method returns the size of vector entries.
	 *
	 * @return Size of vector entries
	 */
	public int getEntriesSize() {
		return entries.size();
	}

	/**
	 * This method returns the vector entries.
	 *
	 * @return Vector entries
	 */
	public Vector<Entry> getEntries() {
		return entries;
	}

	/**
	 * This method returns the enumeration of vector entries.
	 *
	 * @return Enumeration of vector entries
	 */
	public Enumeration<Entry> getElements() {
		return entries.elements();
	}

	/**
	 * This method returns the version of the database.
	 *
	 * @return Version of database
	 */
	public int getVersion() {
		return pdbVersion;
	}

	@SuppressWarnings({"unused", "RedundantSuppression"})
	public int getIterations() {
		return pdbIterations;
	}
	
	// categories -----------------------------------------------------
	/**
	 * This method returns a category name from the vector categories.
	 *
	 * @param category Index of category in vector categories
	 *
	 * @return Category name
	 */
	@SuppressWarnings({"unused", "RedundantSuppression"})
	public String getCategoryName(int category) throws ArrayIndexOutOfBoundsException {
		return categories.get(category);
	}

	/**
	 * This method sets the vector categories to the specified vector.
	 *
	 * @param myCategories New category vector
	 */
	@SuppressWarnings({"unused", "RedundantSuppression"})
	//public void setCategories(Vector<String> myCategories) { // Java 1.5
	public void setCategories(Vector<String> myCategories) {
		categories = myCategories;
	}

	/**
	 * This method returns the vector categories.
	 *
	 * @return Vector categories
	 */
	@SuppressWarnings({"unused", "RedundantSuppression"})
	public Vector<String> getCategories() {
		return categories;
	}

	// loadDataOriginal -----------------------------------------------
	/**
	 * This method loads a Keyring database and generates entry objects for each account.
	 * It is close to the original code from KeyringEditor v1.1 by Markus Griessnig.
	 * It can only handle files up to length BUFFER_SIZE.
	 * It is no longer in use.
	 *
	 * @param iStream Incoming file or content uri
	 * @throws java.lang.Exception with error string
	 */
	@SuppressWarnings({"unused", "RedundantSuppression"})
	public void loadDataOriginal(InputStream iStream) throws Exception {
		final int bufferSize = BUFFER_SIZE;
		int entryLength;
		int pdbLength;
		int emptyTitle = 0;
		int start = 0;
		int len;
		int reallen;
		byte[] iv = null;
		byte[] encrypted = null;
		String title = null;

		// record entry descriptors
		int[] pdbOffset;    // 4
		int[] pdbAttribute; // 1
		int[] pdbUniqueId;  // 3

		// initialisation
		entries.clear();
		categories.clear();

		// read database
		byte[] data = new byte[bufferSize];
		pdbLength = iStream.read(data);

		if(pdbLength == bufferSize) {
			throw new Exception("File too large.");
		}

		iStream.close();

		if(DEBUG) {
			System.out.println("\n========== loadData() original ==========\n");
		}

		// read header
		pdbHeader = sliceBytes(data, 0, 78);
		pdbName = sliceString(data, 0, 32);
		pdbFlags = (int)sliceNumber(data, 32, 2);
		pdbVersion = (int)sliceNumber(data, 34, 2); // 12 byte time information
		pdbModNumber = sliceNumber(data, 48, 4);
		pdbAppInfoOffset = (int)sliceNumber(data, 52, 4);
		pdbSortInfoOffset = (int)sliceNumber(data, 56, 4);
		pdbType = sliceString(data, 60, 4);
		pdbCreator = new String(data, 64, 4); // 8 byte unknown
		pdbNumRecords = (int)sliceNumber(data, 76, 2);

		// keyring declares itself able to handle vCard files so that Dropbox can locate it.
		// check that this is not really a vCard file.
		if(!pdbName.startsWith("Keys-Gtkr") && pdbName.toLowerCase().contains("begin:vcard")) {
			throw new Exception("This is a vCard format file. " +
					"Keyring is unable to open vCard files. Please select an app that can.");
		}

		// check Keyring database format
		if(!(pdbVersion == 4 || pdbVersion == 5)) {
			throw new Exception("Wrong Keyring database format.");
		}

		// offsets
		pdbOffset = new int[pdbNumRecords];
		pdbAttribute = new int[pdbNumRecords];
		pdbUniqueId = new int[pdbNumRecords];

		for(int i=0; i<pdbNumRecords; i++) {
			pdbOffset[i] = (int)sliceNumber(data, 78 + (i*8), 4);
			pdbAttribute[i] = (int)(sliceNumber(data, 78 + 4 + (i*8), 1));
			pdbUniqueId[i] = (int)sliceNumber(data, 78 + 4 + 1 + (i*8), 3);

			if(DEBUG) {
				System.out.println(i + ": " + pdbOffset[i] + " / 0x" +
						Integer.toHexString(pdbAttribute[i]) + " / " + pdbUniqueId[i]);
			}
		}

		if(DEBUG) {
			printPDBHeader();
		}

		pdbCategories = sliceBytes(data, pdbAppInfoOffset, 276);

		// determine the category list
		for(int i=0; i<16; i++) {
			String categoryName = sliceString(data, pdbAppInfoOffset + 2 + (16 * i), 16);

			if (!categoryName.equals("")) {
				categories.add(categoryName);
			}
		}

		if(pdbVersion == 5) {
			if(pdbNumRecords <= 0) {
				throw new Exception("No real data.");
			}

			byte[] salt = sliceBytes(data, pdbAppInfoOffset + 276, 8);
			int iter = (int)sliceNumber(data, pdbAppInfoOffset + 276 + 8, 2);
			int cipher = (int)sliceNumber(data, pdbAppInfoOffset + 276 + 8 + 2, 2);
			byte[] hash = sliceBytes(data, pdbAppInfoOffset + 276 + 8 + 2 + 2, 8);

			// initialize crypto Object
			crypto = new Crypto(null, 5, salt, hash, iter, cipher);

			start = 0; // start with first record

			switch(cipher) {
				case 1: break; // triple des
				case 2: break; // aes 128 bit
				case 3: break; // aes 256 bit
				default: throw new Exception("No cipher not supported.");
			}

//            System.out.println("num iterations = " + iter + "\n");
		}

		if(pdbVersion == 4) {
			if(pdbNumRecords <= 1) { // only password information
				throw new Exception("No real data.");
			}

			recordZeroAttribute = pdbAttribute[0];
			recordZeroUniqueId = pdbUniqueId[0];
			recordZeroLength = pdbOffset[1] - pdbOffset[0];

			// load up password information (entry 0)
			crypto = new Crypto(sliceBytes(data, pdbOffset[0], pdbOffset[1] - pdbOffset[0]), 4);

			start = 1; // start with second record
		}

		// example (Keyring database format 4):
		// numberOfEntries = 4
		// entry 0 = password information
		// entry 1
		// entry 2
		// entry 3

		for(int i=start; i<pdbNumRecords; i++) {

			// check record attribute
			if((pdbAttribute[i] & 0xF0) == 0x40) {
				// determine entry length
				if(i == pdbNumRecords - 1) {
					entryLength = pdbLength - pdbOffset[i];
				}
				else {
					entryLength = pdbOffset[i+1] - pdbOffset[i];
				}

				if(DEBUG) {
				    System.out.println("i=" + i + ": " + pdbOffset[i] + " / " + entryLength);
				}

				if(pdbVersion == 4) { // Keyring database format 4
					// title + \0 + encrypted data
					title = sliceString(data, pdbOffset[i], -1);
					iv = null;
					encrypted = sliceBytes(data, pdbOffset[i] + title.length() + 1, entryLength - title.length() - 1);
				}

				if(pdbVersion == 5) {
					// get length of field
					len = (int)sliceNumber(data, pdbOffset[i], 2);
					reallen = (len + 1) & ~1; // padding for next even address

					title = sliceString(data, pdbOffset[i] + 4, len);

					int ivlen = 8; // tripledes
					if(crypto.type == 2 || crypto.type == 3) ivlen = 16; // aes

					iv = sliceBytes(data, pdbOffset[i] + reallen + 4, ivlen);
					encrypted = sliceBytes(data, pdbOffset[i] + reallen + 4 + ivlen, entryLength - (reallen + 4 + ivlen));
				}

				// Keyring: empty title possible
				if(title != null && title.equals("")) {
					title = "#" + (emptyTitle++);
				}

				// generate entry object
				Entry myEntry = new Entry(
						i,
						title,
						pdbAttribute[i] & 15,
						encrypted,
						crypto,
						pdbAttribute[i],
						pdbUniqueId[i],
						entryLength,
						iv);

				entries.add(myEntry);
			}
		}
	}

	// loadData ------------------------------------------------
	/**
	 * This method loads a Keyring database and generates entries.
	 * It does not limit the length of the database file.
	 *
	 * @param iStream Incoming file or content uri
	 * @throws java.lang.Exception with error string
	 */
	public void loadData(InputStream iStream) throws Exception {
		final int bufferSize = MAX_ENTRY_SIZE;
		int entryLength;
		int bytesRead;
		int emptyTitle = 0;
		int start = 0;
		byte[] iv = null;
		byte[] encrypted = null;
		String title = null;

		// record entry descriptors
		int[] pdbOffset;    // 4
		int[] pdbAttribute; // 1
		int[] pdbUniqueId;  // 3

		// initialisation
		entries.clear();
		categories.clear();

		// read database
		byte[] data = new byte[bufferSize];

		if(DEBUG) {
			System.out.println("\n========== loadData() ==========\n");
		}

		// use a buffered input stream as we now issue a separate read for each entry
		InputStream inStream = new BufferedInputStream(iStream);

		// read header
		bytesRead = inStream.read(data, 0, 78);
		if(bytesRead != 78) {
			throw new Exception("Failed to load file header.");
		}

		pdbHeader = sliceBytes(data, 0, 78);
		pdbName = sliceString(data, 0, 32);
		pdbFlags = (int)sliceNumber(data, 32, 2);
		pdbVersion = (int)sliceNumber(data, 34, 2); // 12 byte time information
		pdbModNumber = sliceNumber(data, 48, 4);
		pdbAppInfoOffset = (int)sliceNumber(data, 52, 4);
		pdbSortInfoOffset = (int)sliceNumber(data, 56, 4);
		pdbType = sliceString(data, 60, 4);
		pdbCreator = new String(data, 64, 4); // 8 byte unknown
		pdbNumRecords = (int)sliceNumber(data, 76, 2);

		// keyring declares itself able to handle vCard files so that Dropbox can locate it.
		// check that this is not really a vCard file.
		if(!pdbName.startsWith("Keys-Gtkr") && pdbName.toLowerCase().contains("begin:vcard")) {
			throw new Exception("This is a vCard format file. " +
					"Keyring is unable to open vCard files. Please select an app that can.");
		}

		// check Keyring database format
		if(!(pdbVersion == 4 || pdbVersion == 5)) {
			throw new Exception("Wrong Keyring database format.");
		}
		if((pdbVersion == 5 && pdbNumRecords <= 0) || (pdbVersion == 4 && pdbNumRecords <= 1)) {
			throw new Exception("No real data.");
		}
		if(pdbAppInfoOffset != 78 + 2 + pdbNumRecords * 8) {
			throw new Exception("Bad app info offset.");
		}

		bytesRead = inStream.read(data, 78, pdbAppInfoOffset - 78 + 276);
		if(bytesRead != pdbAppInfoOffset - 78 + 276) {
			throw new Exception("Failed to read offsets and categories.");
		}

		// offsets
		pdbOffset = new int[pdbNumRecords];
		pdbAttribute = new int[pdbNumRecords];
		pdbUniqueId = new int[pdbNumRecords];

		for(int i=0; i<pdbNumRecords; i++) {
			pdbOffset[i] = (int)sliceNumber(data, 78 + (i*8), 4);
			pdbAttribute[i] = (int)(sliceNumber(data, 78 + 4 + (i*8), 1));
			pdbUniqueId[i] = (int)sliceNumber(data, 78 + 4 + 1 + (i*8), 3);

			if(DEBUG) {
				System.out.println(i + ": " + pdbOffset[i] + " / 0x" +
						Integer.toHexString(pdbAttribute[i]) + " / " + pdbUniqueId[i]);
			}
		}

		if(DEBUG) {
			printPDBHeader();
		}

		pdbCategories = sliceBytes(data, pdbAppInfoOffset, 276);

		// determine the category list
		for(int i=0; i<16; i++) {
			String categoryName = sliceString(data, pdbAppInfoOffset + 2 + (16 * i), 16);

			if (!categoryName.equals("")) {
				categories.add(categoryName);
			}
		}

		if(pdbVersion == 5) {
			if(pdbOffset[0] != pdbAppInfoOffset + 276 + 20) {
				throw new Exception(pdbOffset[0] + " bad first entry offset, version 5.");
			}

			bytesRead = inStream.read(data, pdbAppInfoOffset + 276, 20);
			if(bytesRead != 20) {
				throw new Exception("Failed to read cypher init, version 5.");
			}

			byte[] salt = sliceBytes(data, pdbAppInfoOffset + 276, 8);
			pdbIterations = (int)sliceNumber(data, pdbAppInfoOffset + 276 + 8, 2);
			int cipher = (int)sliceNumber(data, pdbAppInfoOffset + 276 + 8 + 2, 2);
			byte[] hash = sliceBytes(data, pdbAppInfoOffset + 276 + 8 + 2 + 2, 8);

			switch(cipher) {
				case 1: break; // triple des
				case 2: break; // aes 128 bit
				case 3: break; // aes 256 bit
				default: throw new Exception("Cipher " + cipher + " not supported, version 5.");
			}
			if(pdbIterations < 50) {
				throw new Exception(pdbIterations + " bad num iterations, version 5.");
			}

			// initialize crypto Object
			crypto = new Crypto(null, 5, salt, hash, pdbIterations, cipher);

			start = 0; // start with first record
		}

		if(pdbVersion == 4) {
			recordZeroAttribute = pdbAttribute[0];
			recordZeroUniqueId = pdbUniqueId[0];
			recordZeroLength = pdbOffset[1] - pdbOffset[0];

			if(pdbOffset[0] != pdbAppInfoOffset + 276) {
				throw new Exception(pdbOffset[0] + " bad first entry offset, version 4.");
			}
			if(recordZeroLength < Crypto.SALT_SIZE + Crypto.MD5_DIGEST_LENGTH) {
				throw new Exception(recordZeroLength + " record zero too small, version 4.");
			}

			bytesRead = inStream.read(data, 0, recordZeroLength);
			if(bytesRead != recordZeroLength) {
				throw new Exception("Failed to read entry zero in version 4.");
			}

			// load up password information (entry 0)
			crypto = new Crypto(sliceBytes(data, 0, recordZeroLength), 4);

			start = 1; // start with second record
		}

		// example (Keyring database format 4):
		// numberOfEntries = 4
		// entry 0 = password information
		// entry 1
		// entry 2
		// entry 3

		for(int i=start; i<pdbNumRecords; i++) {

			// determine entry length and read entry
			if(i == pdbNumRecords - 1) {
				// final entry, read to end of file
				entryLength = inStream.read(data);
				if(entryLength <= 0) {
					// original Palm Pilot files may have zero length entries
					continue;
				}
				if(entryLength == bufferSize) {
					throw new Exception("Final entry too big.");
				}
			} else {
				entryLength = pdbOffset[i+1] - pdbOffset[i];
				if(entryLength == 0) {
					// original Palm Pilot files may have zero length entries
					continue;
				}
				if(entryLength < 0) {
					throw new Exception("Entry " + i + " has an invalid length.");
				}
				if(entryLength >= bufferSize) {
					throw new Exception(entryLength + ": entry " + i + " too big.");
				}
				bytesRead = inStream.read(data, 0, entryLength);
				if(bytesRead != entryLength) {
					throw new Exception("Failed to read entry number " + i + ".");
				}
			}

			// check record attribute, original Palm Pilot files have many entries marked invalid
			if((pdbAttribute[i] & 0xF0) != 0x40)
				continue;

			if(DEBUG) {
				System.out.println("i=" + i + ": " + pdbOffset[i] + " / " + entryLength);
			}

			if(pdbVersion == 4) { // Keyring database format 4
				// title + \0 + encrypted data
				title = sliceString(data, 0, -1);
				if(entryLength <= title.length() + 1) {
					throw new Exception("Entry " + i + " is empty, version 4.");
				}
				iv = null;
				encrypted = sliceBytes(data, title.length() + 1, entryLength - title.length() - 1);
			}

			if(pdbVersion == 5) {
				// get length of field
				int len = (int)sliceNumber(data, 0, 2);
				int reallen = (len + 1) & ~1; // padding for next even address

				title = sliceString(data, 4, len);

				int ivlen = 8; // tripledes
				if(crypto.type == 2 || crypto.type == 3) ivlen = 16; // aes

				if(entryLength <= reallen + 4 + ivlen) {
					throw new Exception("Entry " + i + " is empty, test version 5.");
				}

				iv = sliceBytes(data, reallen + 4, ivlen);
				encrypted = sliceBytes(data, reallen + 4 + ivlen, entryLength - (reallen + 4 + ivlen));
			}

			// Keyring: empty title possible
			if(title != null && title.equals("")) {
				title = "#" + (emptyTitle++);
			}

			// generate entry object
			Entry myEntry = new Entry(
				i,
				title,
				pdbAttribute[i] & 15,
				encrypted,
				crypto,
				pdbAttribute[i],
				pdbUniqueId[i],
				entryLength,
				iv);

			entries.add(myEntry);
		}
		inStream.close();
	}


	// saveData -------------------------------------------------------
	/**
	 * This method checks the record lengths before the file is opened for writing.
	 */
	public void checkRecordLengths() throws Exception {
		for(Enumeration<Entry> e = entries.elements(); e.hasMoreElements(); ) {
			Entry entry = e.nextElement();
			byte[] encodedTitle;
			int recordLen = 0;
			switch(pdbVersion) {
				case 4:
					encodedTitle = stringToByteArray(entry.getTitle());
					recordLen = encodedTitle.length + entry.encrypted.length + 1;
					break;
				case 5:
					encodedTitle = convertStringToField(entry.getTitle(), 0);
					recordLen = encodedTitle.length + entry.iv.length + entry.encrypted.length;
					break;
			}

			if (recordLen != entry.recordLength) {
				throw new Exception("Record '" + entry.getTitle() + "' length " +
						recordLen + " does not match stored length " + entry.recordLength +
						".\nThe database file has not been modified.");
			}
		}
	}

	/**
	 * This method calls the saveData method according to database version (pdbVersion).
	 *
	 * @param outStream Keyring database
	 */
	public void saveData(OutputStream outStream) throws Exception {
		if(DEBUG) {
			 System.out.println("saveData");
		}

		switch(pdbVersion) {
			case 4: saveData_4(outStream); break;
			case 5: saveData_5(outStream); break;
		}
	}

	/**
	 * This method saves all entries in the specified database (Database format 4).
	 *
	 * @param fp Keyring database
	 */
	public void saveData_4(OutputStream fp) throws Exception {
		pdbAppInfoOffset = 78 + 8 * entries.size() + 2 + 8; // + 8 for recordZero
		pdbNumRecords = entries.size() + 1; // + 1 for recordZero
		int offset = pdbAppInfoOffset + 276;

		// write header
		fp.write(pdbHeader, 0, 52);
		fp.write(numberToByte(pdbAppInfoOffset, 4), 0, 4);
		fp.write(pdbHeader, 56, 20);
		fp.write(numberToByte(pdbNumRecords, 2), 0, 2);

		// write offset recordZero
		fp.write(numberToByte(offset, 4), 0, 4);
		fp.write(numberToByte(recordZeroAttribute, 1), 0, 1);
		fp.write(numberToByte(recordZeroUniqueId, 3), 0, 3);

		offset += recordZeroLength;

		// write offsets
		for(Enumeration<Entry> e = entries.elements(); e.hasMoreElements(); ) {
			Entry entry = e.nextElement();

			fp.write(numberToByte(offset, 4), 0, 4);
			fp.write(numberToByte(entry.attribute, 1), 0, 1); // category
			fp.write(numberToByte(entry.uniqueId, 3), 0, 3);

			if(DEBUG) {
				System.out.println("saveData4: " + offset + ", " + entry.attribute + ", " + entry.uniqueId);
			}

			offset += entry.recordLength;
		}

		fp.write(0x0000);
		fp.write(0x0000);

		// write categories
		// (no need to update categories they are only editable in KeyringEditor)
		fp.write(pdbCategories, 0, 276);

		// write password information
		fp.write(crypto.recordZero);

		// write records
		for(Enumeration<Entry> e = entries.elements(); e.hasMoreElements(); ) {
			Entry entry = e.nextElement();

			fp.write(stringToByteArray(entry.getTitle()));
			fp.write(0x00);
			fp.write(entry.encrypted);
		}

		fp.close();
	}

	/**
	 * This method saves all entries in the specified database (Database format 5).
	 *
	 * @param fp Keyring database
	 */
	public void saveData_5(OutputStream fp) throws Exception {
		pdbNumRecords = entries.size();
		pdbAppInfoOffset = 78 + 8 * pdbNumRecords + 2;
		int offset = pdbAppInfoOffset + 276 + 20; // salt hash type

		// write header
		fp.write(pdbHeader, 0, 52);
		fp.write(numberToByte(pdbAppInfoOffset, 4), 0, 4);
		fp.write(pdbHeader, 56, 20);
		fp.write(numberToByte(pdbNumRecords, 2), 0, 2);

		// write offsets
		for(Enumeration<Entry> e = entries.elements(); e.hasMoreElements(); ) {
			Entry entry = e.nextElement();

			fp.write(numberToByte(offset, 4), 0, 4);
			fp.write(numberToByte(entry.attribute, 1), 0, 1); // category
			fp.write(numberToByte(entry.uniqueId, 3), 0, 3);

			offset += entry.recordLength;
		}

		fp.write(0x0000);
		fp.write(0x0000);

		// write categories
		// (no need to update categories they are only editable in KeyringEditor)
		fp.write(pdbCategories, 0, 276);

		// write SALT HASH TYPE (db_format.txt)
		fp.write(crypto.salt);
		fp.write(numberToByte(crypto.iter, 2));
		fp.write(numberToByte(crypto.type, 2));
		fp.write(crypto.hash);

		// write records
		for(Enumeration<Entry> e = entries.elements(); e.hasMoreElements(); ) {
			Entry entry = e.nextElement();

			if(DEBUG) {
				System.out.println("Write entry: " + entry.getTitle());
			}
			fp.write(convertStringToField(entry.getTitle(), 0));
			fp.write(entry.iv);
			fp.write(entry.encrypted);
		}

		fp.close();
	}

	// toRecordFormat4 ------------------------------------------------
	/**
	 * This method adds todays date (DateType format) to decrypted data (for database format 4).
	 *
	 * @param data Example: Account + \0 + Password + \0 + Notes + \0
	 *
	 * @return data + todays date in datetype format
	 */
	public static byte[] toRecordFormat4(String data) {
		byte[] today = getDateType();
		byte[] buffer = stringToByteArray(data);
		byte[] result = new byte[buffer.length + 2];

		System.arraycopy(buffer, 0, result, 0, buffer.length);
		result[buffer.length] = today[1];
		result[buffer.length + 1] = today[0];

		return result;
	}

	// toRecordFormat5 ------------------------------------------------
	/**
	 * This method adds todays date (DateType format) to decrypted data (for database format 5).
	 *
	 * @param account Entry account
	 * @param password Entry password
	 * @param notes Entry notes
	 *
	 * @return decrypted data in database format 5
	 */
	public static byte[] toRecordFormat5(String account, String password, String notes) {
		// Format:
		// field (account)
		// field (password)
		// field (notes)
		// field (datetype)
		// 0xff
		// 0xff
		// random padding to multiple of 8 bytes
		byte[] datetype = {0x00, 0x02, 0x03, 0x00, 0x00, 0x00};

		byte[] field1 = account.getBytes();
		byte[] field2 = password.getBytes();
		byte[] field3 = notes.getBytes();

		int lenField1 = field1.length;
		int lenField2 = field2.length;
		int lenField3 = field3.length;

		if(lenField1 != 0) {
			field1 = convertStringToField(account, 1);
			lenField1 = field1.length;
		}

		if(lenField2 != 0) {
			field2 = convertStringToField(password, 2);
			lenField2 = field2.length;
		}

		if(lenField3 != 0) {
			field3 = convertStringToField(notes, 255);
			lenField3 = field3.length;
		}

		byte[] now = getDateType();
		datetype[4] = now[1];
		datetype[5] = now[0];

		int padding = (lenField1 + lenField2 + lenField3 + 6 + 2) % 8;
		byte[] result = new byte[lenField1 + lenField2 + lenField3 + 6 + 2 + padding];
		Arrays.fill(result, (byte)0xff);

		if(lenField1 != 0) {
			System.arraycopy(field1, 0, result, 0, lenField1);
		}

		if(lenField2 != 0) {
			System.arraycopy(field2, 0, result, lenField1, lenField2);
		}

		if(lenField3 != 0) {
			System.arraycopy(field3, 0, result, lenField1 + lenField2, lenField3);
		}

		System.arraycopy(datetype, 0, result, lenField1 + lenField2 + lenField3, 6);

		return result;
	}

	/**
	 * This method converts a string in the format used by database format 5 (Field).
	 *
	 * @param field Text
	 * @param label Label information (account=1, password=2, notes=255)
	 *
	 * @return Field
	 */
	public static byte[] convertStringToField(String field, int label) {
		// Format:
		// 2 byte length of field
		// 1 byte label
		// 1 byte 0x00
		// data
		// 0/1 padding for next even address
		byte[] buffer = stringToByteArray(field);
		int padding = 0;
		int len = buffer.length;

		if((len % 2) == 1) {
			padding = 1;
		}

		byte[] result = new byte[4 + len + padding];
		Arrays.fill(result, (byte)0);

		System.arraycopy(numberToByte(len,2), 0, result, 0, 2);
		System.arraycopy(numberToByte(label,1), 0, result, 2, 1);
		result[3] = (byte)0x00;
		System.arraycopy(buffer, 0, result, 4, len);

		return result;
	}

	/**
	 * Converts a string to a byte array using the specified character encoding
	 *
	 * @param data String to convert
	 * @return byte array
	 */
	public static byte[] stringToByteArray(String data) {
		byte[] buffer;
		try {
			buffer = data.getBytes(usingCharset);
		} catch (UnsupportedEncodingException e) {
			// (can't happen because we checked charset is supported in constructor)
			buffer = data.getBytes();
		}
		return buffer;
	}


	// getNewUniqueId -------------------------------------------------
	/**
	 * This method searches the entries for the highest id.
	 *
	 * @return New unique id
	 */
	public int getNewUniqueId() {
		int id = 0;

		for(Enumeration<Entry> e = entries.elements(); e.hasMoreElements(); ) {
			
			Entry entry = e.nextElement();

			if(entry.getUniqueId() > id) {
				id = entry.getUniqueId();
			}
		}
		
		id = id + 1;
		
		return(id);
	}

	// writeNewDatabase -----------------------------------------------
	/**
	 * This method creates a minimal version 5 database.
	 */
	public void initNewDatabase() {
		int[] header = {
		0x4B, 0x65, 0x79, 0x73, 0x2D, 0x47, 0x74, 0x6B, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x4B, 0x65, 0x79, 0x72, 0x69, 0x6E, 0x67, 0x45, 0x64, 0x69, 0x74, 0x6F, 0x72, 0x31, 0x2E, 0x32,
		0x00, 0x08, 0x00, 0x05, 0xBD, 0xDB, 0x65, 0x06, 0xBD, 0xDB, 0x65, 0x0D, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x00, 0x47, 0x6B, 0x79, 0x72,
		0x47, 0x74, 0x6B, 0x72, 0x00, 0xB7, 0x30, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01};

		int[] saltData = {0xB0, 0xDA, 0x43, 0x4A, 0xB0, 0xDA, 0x43, 0x4A};

		// hash of the passwd 'test' for testing
		int[] hashData = {0xD3, 0x31, 0x73, 0x81, 0x43, 0x7D, 0x1A, 0x4B};

		entries.clear();
		categories.clear();

		// write header
		for (int i = 0; i < pdbHeader.length; i++) {
			pdbHeader[i] = (byte)header[i];
		}

		pdbHeader = sliceBytes(pdbHeader, 0, 78);
		pdbName = sliceString(pdbHeader, 0, 32);
		pdbFlags = (int)sliceNumber(pdbHeader, 32, 2);
		pdbVersion = (int)sliceNumber(pdbHeader, 34, 2); // 12 byte time information
		pdbModNumber = sliceNumber(pdbHeader, 48, 4);
		pdbAppInfoOffset = (int)sliceNumber(pdbHeader, 52, 4);
		pdbSortInfoOffset = (int)sliceNumber(pdbHeader, 56, 4);
		pdbType = sliceString(pdbHeader, 60, 4);
		pdbCreator = new String(pdbHeader, 64, 4); // 8 byte unknown
		pdbNumRecords = (int)sliceNumber(pdbHeader, 76, 2);

		if (DEBUG) {
			System.out.println("new database: version " + pdbVersion + ", offset " + pdbAppInfoOffset +
					", num records " + pdbNumRecords);
		}

		// write category names
		byte[] cat = new byte[256];
		byte[] cat1 = ("no category").getBytes();
		Arrays.fill(cat, (byte) 0x00);
		System.arraycopy(cat1, 0, cat, 0, cat1.length);
		// write renamed categories
		pdbCategories[0] = (byte) 0x1F;
		pdbCategories[1] = (byte) 0x1F;
		// write category titles
		System.arraycopy(cat, 0, pdbCategories, 2, cat.length);
		// write unique ids
		for (int i = 0; i < 16; i++) {
			pdbCategories[258 + i] = (byte) i;
		}
		// write padding
		pdbCategories[274] = (byte) 0x0F;
		pdbCategories[275] = (byte) 0x00;

		// initialize crypto Object
		byte[] salt = new byte[8];
		byte[] hash = new byte[8];
		pdbIterations = 10000;
		for (int i = 0; i < 8; i++) {
			salt[i] = (byte) saltData[i];
			hash[i] = (byte) hashData[i];
		}
		crypto = new Crypto(null, 5, salt, hash, pdbIterations, 2);
	}

	// writeNewDatabase -----------------------------------------------
	/**
	 * This method dumps a minimal version 4 database with password "test".
	 *
	 * @param filename New database filename
	 */
	@SuppressWarnings({"unused", "RedundantSuppression"})
	public static void writeNewDatabase(String filename) {
		int[] header = {
				0x4B, 0x65, 0x79, 0x73, 0x2D, 0x47, 0x74, 0x6B, 0x72, 0x00,
				0x6B, 0x72, 0x5F, 0x61, 0x70, 0x70, 0x6C, 0x5F, 0x61, 0x36,
				0x38, 0x6B, 0x00, 0x00, 0x73, 0x79, 0x73, 0x70, 0x04, 0x00,
				0x73, 0x70, 0x00, 0x08, 0x00, 0x04, 0xBD, 0xDB, 0x65, 0x06,
				0xBD, 0xDB, 0x65, 0x0D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x0E, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00,
				0x47, 0x6B, 0x79, 0x72, 0x47, 0x74, 0x6B, 0x72, 0x00, 0xB7,
				0x30, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
				0x01, 0x74, 0x50, 0xB7, 0x30, 0x01, 0x00, 0x00, 0x01, 0x88,
				0x40, 0xB7, 0x30, 0x02, 0x00, 0x00, 0x1F, 0x1F};

		int[] data = {
				0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
				0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x0F, 0x00, 0xB0, 0xDA,
				0x43, 0x4A, 0x91, 0x55, 0x12, 0xEC, 0xD5, 0x96, 0xCD, 0x21,
				0x9A, 0xFC, 0x2D, 0x01, 0x9C, 0x2F, 0xC7, 0x13, 0x61, 0x00,
				0xF0, 0x3B, 0x16, 0xCC, 0x25, 0xCF, 0x49, 0xC0};

		int i;
		File db;
		FileOutputStream fp;
		byte[] cat = new byte[256];
		byte[] cat1 = ("no category").getBytes();

		// open new database
		try {
			db = new File(filename);
			fp = new FileOutputStream(db);

			// write header
			for(i=0; i<header.length; i++) {
				fp.write((byte)header[i]);
			}

			// write category-names
			Arrays.fill(cat, (byte)0x00);
			System.arraycopy(cat1, 0, cat, 0, cat1.length);
			fp.write(cat, 0, 256);

			// write password information and record 1
			for(i=0; i<data.length; i++) {
				fp.write((byte)data[i]);
			}

			fp.close();
		}
		catch(Exception e) {
			System.err.println("Caught Exception: " + e.getMessage());
		}
	}

	// tools ----------------------------------------------------------
	/**
	 * This method converts a long into a byte array.
	 *
	 * @param number Number
	 * @param len Size of byte array
	 *
	 * @return Byte array representation of number
	 */
	public static byte[] numberToByte(long number, int len) {
		int i, shift;
		byte[] buffer = new byte[len];

		for(i=0, shift=((len-1) * 8); i<len; i++, shift -= 8) {
			buffer[i] = (byte)(0xFF & (number >> shift));
		}

		return buffer;
	}

	/**
	 * This method converts a byte to int.
	 *
	 * @param b Byte
	 *
	 * @return Int representation of Byte
	 */
	public static int unsignedByteToInt(byte b) {
		return b & 0xFF;
	}

	/**
	 * This method slices a byte array from an byte array.
	 *
	 * @param data Byte array
	 * @param start Index to start from
	 * @param length Length of byte array to slice out
	 *
	 * @return Byte array
	 */
	public static byte[] sliceBytes(byte[] data, int start, int length) {
		byte[] bytes = new byte[length];

		if (length >= 0) System.arraycopy(data, start, bytes, 0, length);

		return bytes;
	}

	/**
	 * This method slices a byte array from an byte array and converts it to a long.
	 *
	 * @param data Byte array
	 * @param start Index to start from
	 * @param length Length of byte array to slice out
	 *
	 * @return Long representation of the byte array
	 */
	public static long sliceNumber(byte[] data, int start, int length) {
		long value = 0, factor = 1;

		for(int i=0; i<length; i++) {
			value += unsignedByteToInt(data[start + length - (i + 1)]) * factor;
			factor *= 256;
		}

		return value;
	}

	/**
	 * This method slices a byte array from an byte array and converts it to a string.
	 *
	 * @param data Byte array
	 * @param start Index to start from
	 * @param length Length of byte array to slice out
	 *
	 * @return String representation of the byte array
	 */
	public static String sliceString(byte[] data, int start, int length) {
		int realLength = 0;

		if(length == -1) {
			// no specific max length (make it to the end of the array)
			length = data.length - start;
		}

		while(realLength < length && data[start + realLength] != 0) {
			realLength++;
		}

		try {
			return new String(data, start, realLength, usingCharset);
		} catch (UnsupportedEncodingException e) {
			// (can't happen because we checked charset is supported in constructor)
			return new String(data, start, realLength);
		}
	}


	//pn: pbkdf2 algorithm test
	//Test vectors from RFC 3962 -- all tests pass except if
	//password exceeds block size (64 chars) -- but password is limited to 40 chars
	public String testPbkdf2(String passwdStr, String saltStr, int iter, int keylen) {
		char[] hexNumbers = {'0','1','2','3','4','5','6','7','8','9','a','b','c','d','e','f'};
		byte[] passwd = passwdStr.getBytes();
		byte[] salt = saltStr.getBytes();
		try {
			byte[] key = crypto.pbkdf2(passwd, salt, iter, keylen);
			StringBuilder str = new StringBuilder();
			for (int i = 0; i < 32; i++) {
				str.append(hexNumbers[(key[i] & 0xFF) / 16]);
				str.append(hexNumbers[(key[i] & 0xFF) % 16]);
				str.append(" ");
			}
			return str.toString();
		}
		catch (Exception ex) {
			return "testPbkdf2 exception: " + ex.getMessage();
		}
	}
	
	
	// ----------------------------------------------------------------
	// private --------------------------------------------------------
	// ----------------------------------------------------------------

	/**
	 * debug use only
	 */
	private void printPDBHeader() {
		System.out.println("PDB Name: " + pdbName);
		System.out.println("PDB Flags: " + pdbFlags);
		System.out.println("PDB Version: " + pdbVersion);
		System.out.println("PDB Modification Number: " + pdbModNumber);
		System.out.println("PDB AppInfoOffset: " + pdbAppInfoOffset);
		System.out.println("PDB SortInfoOffset: " + pdbSortInfoOffset);
		System.out.println("PDB Type: " + pdbType);
		System.out.println("PDB Creator: " + pdbCreator);
		System.out.println("PDB NumberOfRecords: " + pdbNumRecords + "\n");
	}


	// DateType -------------------------------------------------------
	/**
	 * This method return todays date in DateType format.
	 *
	 * @return Todays date in DateType format (byte[2])
	 */
	private static byte[] getDateType() {
		int day, month, year;
		int[] intResult = new int[2];
		byte[] byteResult = new byte[2];

		Calendar rightNow = new GregorianCalendar();

		day = rightNow.get(Calendar.DAY_OF_MONTH);
		month = rightNow.get(Calendar.MONTH) + 1; // Calender month from 0 to 11
		year = rightNow.get(Calendar.YEAR) - 1904; // DateType year since 1904

		day = (day & 0x1F); // 5 bit
		month = (month & 0x0F); // 4 bit
		year = (year & 0x7F); // 7 bit

		// DateType (2 bytes): 7 bit year, 4 bit month, 5 bit day
		intResult[0] = day | ((month & 0x07) << 5);
		intResult[1] = (year << 1) | ((month & 0x08) >> 3);

		// System.out.println(intResult[1] + " " + intResult[0]);
		byteResult[0] = (byte)intResult[0];
		byteResult[1] = (byte)intResult[1];

		return byteResult;
	}
}
