/*
Palm Keyring for Android

Copyright (C) 2022 Peter Newman <pn@pnewman.com>

Palm Keyring for Android is based on:
KeyringEditor v1.1
Copyright 2004 Markus Griessnig
Vienna University of Technology Institute of Computer Technology

KeyringEditor is based on:
Java Keyring v0.6
Copyright 2004 Frank Taylor <keyring@lieder.me.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE.  See the GNU General Public License for more details:
<http://www.gnu.org/licenses/>.
*/

// Editor.java

// 25.11.2004

// 01.12.2004: Keyring database format 5 support added
// 05.12.2004: MenuItem Tools - Convert database added
// 12.01.2004: Model.writeNewDatabase() in main() added
// 24.05.2005: showEntry - check no category
// 12.06.2021: add warning that version 4 file format is no longer secure
// 08.01.2022: explicitly truncate the file for api 29 by opening with mode 'wt'
// 08.01.2022: linkify urls to open in default browser when touched
// 06.08.2022: add local file access via Android Storage Access Framework to files menu
// 06.08.2022: combine local and private (legacy) files in current file persistent state
// 06.08.2022: display file name for files from content server
// 20.08.2022: add create new local database file
// 27.08.2022: open file with single entry in show card mode not in list mode
// 02.09.2022: add filename to password dialog
// 30.04.2023: toast limited to two lines in api 31

package com.pnewman.keyring;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.app.DialogFragment;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.database.Cursor;
import android.graphics.Typeface;
import android.net.Uri;
import android.os.Bundle;
import android.os.Environment;
import android.os.Handler;
import android.provider.OpenableColumns;
import android.support.annotation.NonNull;
import android.text.method.ScrollingMovementMethod;
import android.text.util.Linkify;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.View.OnKeyListener;
import android.view.Window;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;
import android.widget.Toast;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;


public class Keyring extends Activity {

    private static final int ACTIVITY_GET_FILENAME = 2;
	private static final int ACTIVITY_CREATE_NEW_SAF_FILE = 3;
	private static final int ACTIVITY_OPEN_SAF_FILE = 4;
    private static final int ACTIVITY_DISPLAY_LIST = 5;
    private static final int ACTIVITY_SET_PREFERENCES = 6;
    private static final int ACTIVITY_NEW_ENTRY = 7;
    private static final int ACTIVITY_EDIT_ENTRY = 8;
    private static final int STATE_INIT = 0;
    private static final int STATE_IDLE = 1;
    private static final int STATE_FILE_LOADED = 2;
    private static final int STATE_READY = 3;
    private static final int STATE_LOCKED = 4;
    public static final String KEY_FILE_NAME = "fileName";
	public static final String KEY_DISPLAY_LIST = "displayList";
    public static final String KEY_DISPLAY_PAGE_TITLE = "displayPageTitle";
    public static final String KEY_DISPLAY_HELP_FILE = "displayHelpFile";
    public static final String KEY_ENTRY_ID = "entryId";
    public static final String KEY_ENTRY_TITLE_TEXT = "entryTitleText";
    public static final String KEY_ENTRY_ACCOUNT_TEXT = "entryAccountText";
    public static final String KEY_ENTRY_PASSWD_TEXT = "entryPasswdText";
    public static final String KEY_ENTRY_NOTES_TEXT = "entryNotesText";
    public static final String KEY_ENTRY_TEXT_SIZE = "entryTextSize";
    public static final String KEY_LIST_SELECTION = "listSelection";
    public static final String KEY_SETTINGS_FILENAME = "filename";
    public static final String KEY_SETTINGS_FONT = "textFont";
    public static final String KEY_SETTINGS_STYLE = "textStyle";
    public static final String KEY_SETTINGS_SIZE = "textSize";
    public static final String KEY_SETTINGS_TIMEOUT = "timeoutValue";
    public static final String KEY_SETTINGS_READ_ONLY_MODE = "readOnlyMode";

    private static final int TIMEOUT_30_SECS = 30000;
    private static final int TIMEOUT_1_MIN = 60000;
    private static final int TIMEOUT_5_MISN = 300000;
    private static final int TIMEOUT_10_MINS = 600000;
    private static final int TIMEOUT_15_MINS = 900000;

	/**
	 * Current loaded keyring database
	 */
	private String mFilename;
	private Uri mContentUri;

	private TextView mBodyText;
	private Handler mHandler;			//one-shot timer
	private Runnable mTimeoutCallback;	//timer callback
	private Button mLockButton;
	private Model model;
	private int mCurrentEntry;
	private ArrayList<String> mTitleList;
	private ArrayList<Integer> mTitleIdList;
	private int mState;
	private long mStartLastTimeout;
	private int mTimeoutValue = TIMEOUT_30_SECS;
	private boolean mReadOnly;
	private Uri mTempContentUri = null;   // temp because Android dialogs are not modal


	@Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

		// prevent screenshot when app goes into background
		getWindow().setFlags(WindowManager.LayoutParams.FLAG_SECURE, WindowManager.LayoutParams.FLAG_SECURE);

        setContentView(R.layout.main);

        mBodyText = findViewById(R.id.body);
    	mBodyText.setMovementMethod(new ScrollingMovementMethod());
    	mBodyText.setClickable(false);		//to avoid dimming on scroll
    	mBodyText.setLongClickable(false);	//to avoid dimming on scroll
		mBodyText.setAutoLinkMask(Linkify.WEB_URLS);  //activate urls
    	mLockButton = findViewById(R.id.button_lock);
    	mTitleList = new ArrayList<>();
    	mTitleIdList = new ArrayList<>();
    	mHandler = new Handler();
    	mFilename = null;
		mContentUri = null;
    	model = null;
    	mCurrentEntry = -1;
    	mStartLastTimeout = 0;
    	mState = STATE_INIT;
    	mReadOnly = true;
    	
    	//hides soft keyboard on startup
    	getWindow().setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_STATE_ALWAYS_HIDDEN);

		//restore previous state, if any
		//typically occurs if device rotated portrait to landscape or back
		if (savedInstanceState != null) {
			mCurrentEntry = savedInstanceState.getInt(KEY_ENTRY_ID);
		}

    	try {
			//restore text display attributes
	        restorePreferences();

	        //check whether intent specifies the file
			boolean contentUri = false;
			String filePath = null;
			Uri uri = getIntent().getData();
			if (uri != null) {
				String scheme = uri.getScheme();
				if (scheme != null && scheme.equals("file")) {
					filePath = uri.getEncodedPath();
				}
				else if (scheme != null && scheme.equals("content")) {
					contentUri = true;
				}
			}

			if (filePath == null && !contentUri) {
				//intent does not specify file so restore previous file
				SharedPreferences settings = getPreferences(0);
				String fileStr = settings.getString(KEY_SETTINGS_FILENAME, null);
				if (fileStr != null && fileStr.length() > 0) {
					uri = Uri.parse(fileStr);
					String scheme = null;
					if (uri != null)
						scheme = uri.getScheme();
					if (scheme == null) {
						// string is a filename
						filePath = fileStr;
					}
					else if (scheme.equals("content")) {
						// string is a content uri
						contentUri = true;
					}
				}
			}

			if (contentUri) {
				// load database from content uri
				loadDatabase(null, uri, false);
			}
			else if (filePath != null && (mFilename == null || mFilename.compareTo(filePath) != 0)) {
				// load database from file
				loadDatabase(filePath, null, false);
			}
    	}
    	catch (ClassCastException e) {
    		//ignore exceptions
    	}

    	//set callback for timer
    	mTimeoutCallback = new Runnable() {
    		public void run() {
    			//lock database on timeout
    			lockDatabase(null);
    		}
    	};

    	if (mState == STATE_INIT) gotoStateIdle();
	}


	//save state, typically occurs if device rotated portrait to landscape or back
    @Override
    protected void onSaveInstanceState(@NonNull Bundle outState) {
        super.onSaveInstanceState(outState);
        outState.putInt(KEY_ENTRY_ID, mCurrentEntry);
    }


    @Override
    protected void onResume() {
        super.onResume();
		if (System.currentTimeMillis() - mStartLastTimeout > mTimeoutValue &&
				mState == STATE_READY) {
			//timeout has expired, lock database
			lockDatabase(null);
		}
    }

	//set menu items accessed from the menu
	@Override
	public boolean onCreateOptionsMenu(Menu menu) {
		MenuInflater inflater = getMenuInflater();
		inflater.inflate(R.menu.keyring_menu, menu);
		return true;
	}


    //respond to menu items selected from the menu
    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
    	int itemId = item.getItemId();
		if (mState == STATE_FILE_LOADED) {
			gotoStateIdle();
		}
    	if (itemId == R.id.menu_open_file) {
			openFileDialog();
			return true;
		}
		if (itemId == R.id.menu_new_saf_file) {
			createNewSafFileDialog();
			return true;
		}
		if (itemId == R.id.menu_open_saf_file) {
			openSafFile();
			return true;
		}
		if (itemId == R.id.menu_new_entry) {
			newEntry();
			return true;
		}
		if (itemId == R.id.menu_edit_entry) {
			editEntry();
			return true;
		}
		if (itemId == R.id.menu_delete_entry) {
			deleteEntryDialog();
			return true;
		}
		if (itemId == R.id.menu_preferences) {
			startActivityForResult(new Intent(this, Preferences.class), ACTIVITY_SET_PREFERENCES);
			return true;
		}
		if (itemId == R.id.menu_about) {
			displayAbout();
			return true;
		}
		if (itemId == R.id.menu_help) {
			loadHelpFile();
			return true;
		}
        return super.onOptionsItemSelected(item);
    }

    
	private void restorePreferences() {
		SharedPreferences settings = getPreferences(0);
	    int fontSetting = settings.getInt(KEY_SETTINGS_FONT, 0);
	    if (fontSetting < 0 || fontSetting >= Preferences.NUM_FONTS) fontSetting = 0;	    
	    Typeface tf = Typeface.DEFAULT;
        switch (fontSetting) {
        	case 0: tf = Typeface.DEFAULT;
        		break;
        	case 1: tf = Typeface.MONOSPACE;
        		break;
        	case 2: tf = Typeface.SANS_SERIF;
        		break;
        	case 3: tf = Typeface.SERIF;
        		break; 	
        }
	    int styleSetting = settings.getInt(KEY_SETTINGS_STYLE, 0);
	    if (styleSetting < 0 || styleSetting >= Preferences.NUM_STYLES) styleSetting = 0;
        mBodyText.setTypeface(tf, styleSetting);

	    int sizeSetting = settings.getInt(KEY_SETTINGS_SIZE, Preferences.DEFAULT_TEXT_SIZE);
	    if (sizeSetting < Preferences.SIZE_START_VALUE ||
	    	sizeSetting > Preferences.SIZE_ARRAY_MAX * 2 + Preferences.SIZE_START_VALUE)
	    	sizeSetting = Preferences.DEFAULT_TEXT_SIZE;
        mBodyText.setTextSize((float) sizeSetting);

        int oldTimeoutValue = mTimeoutValue;
        int timeoutSetting = settings.getInt(KEY_SETTINGS_TIMEOUT, 3);
	    if (timeoutSetting < 0 || timeoutSetting >= Preferences.NUM_TIMEOUT_VALUES) timeoutSetting = 0;
        switch (timeoutSetting) {
        	case 0: mTimeoutValue = TIMEOUT_30_SECS;
        		break;
        	case 1: mTimeoutValue = TIMEOUT_1_MIN;
        		break;
        	case 2: mTimeoutValue = TIMEOUT_5_MISN;
        		break;
        	case 3: mTimeoutValue = TIMEOUT_10_MINS;
        		break; 	
        	case 4: mTimeoutValue = TIMEOUT_15_MINS;
        		break; 	
        }     
		if (mState == STATE_READY && oldTimeoutValue != mTimeoutValue) {
			//timeout value has changed
			resetTimeout();
		}

        mReadOnly =
            settings.getBoolean(KEY_SETTINGS_READ_ONLY_MODE, Preferences.DEFAULT_READ_ONLY_MODE);
	}


	//build password dialog
	public static class PasswdDialogFragment extends DialogFragment {

		public enum passwdDialogType {SET_PASSWD_DIALOG, NEW_DATABASE_PASSWD_DIALOG}
		private static passwdDialogType dialogType = passwdDialogType.SET_PASSWD_DIALOG;
		private static String filename = "";

		public static PasswdDialogFragment newInstance(passwdDialogType type, String fileStr) {
			dialogType = type;
			filename = fileStr;
			return new PasswdDialogFragment();
		}

		@Override
		public Dialog onCreateDialog(Bundle savedInstanceState) {
			LayoutInflater inflater = getActivity().getLayoutInflater();
			final View layout = inflater.inflate(R.layout.passwd_dialog, null);
			final EditText editPasswd = layout.findViewById(R.id.EditText_Passwd);
			final TextView passwdText = layout.findViewById(R.id.TextView_Passwd);

			AlertDialog.Builder builder = new AlertDialog.Builder(getActivity());
			if (dialogType == passwdDialogType.NEW_DATABASE_PASSWD_DIALOG)
				builder.setTitle(R.string.new_database_passwd_dialog_title);
			else
				builder.setTitle(R.string.passwd_dialog_title);
			if (filename != null && filename.length() > 0)
				passwdText.setText("File: " + filename + "\nPassword:");
			else
				passwdText.setText(R.string.passwd_dialog);
			builder.setView(layout);

			//listen for enter key press in the edit window
			editPasswd.setOnKeyListener(new OnKeyListener() {
				public boolean onKey(View v, int keyCode, KeyEvent event) {
					if (event.getAction() == KeyEvent.ACTION_DOWN) {
						if (keyCode == KeyEvent.KEYCODE_ENTER) {
							if (dialogType == passwdDialogType.NEW_DATABASE_PASSWD_DIALOG)
								((Keyring)getActivity()).createNewDatabase(editPasswd.getText());
							else
								((Keyring)getActivity()).checkPassword(editPasswd.getText());
							editPasswd.getText().clear();
							dismiss();
							return true;
						}
					}
					return false;
				}
			});

			builder.setNegativeButton(android.R.string.cancel, new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int id) {
					editPasswd.getText().clear();
					if (dialogType == passwdDialogType.SET_PASSWD_DIALOG &&
							((Keyring)getActivity()).mState != STATE_LOCKED) {
						((Keyring)getActivity()).gotoStateIdle();
					}
				}
			});

			builder.setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int id) {
					if (dialogType == passwdDialogType.NEW_DATABASE_PASSWD_DIALOG)
						((Keyring)getActivity()).createNewDatabase(editPasswd.getText());
					else
						((Keyring)getActivity()).checkPassword(editPasswd.getText());
					editPasswd.getText().clear();
				}
			});

			AlertDialog dialog = builder.create();
			Window window = dialog.getWindow();
			if (window != null) window.setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_STATE_VISIBLE);
			editPasswd.requestFocus();
			return dialog;
		}
	}

	// show passwd dialog on opening file
	public void showPasswdDialog() {
		DialogFragment newDialogFragment =
				PasswdDialogFragment.newInstance(PasswdDialogFragment.passwdDialogType.SET_PASSWD_DIALOG,
						getFileShortDisplayName());
		newDialogFragment.show(getFragmentManager(), "passwdDialogFragment");
	}

	// show passwd dialog to create new database
	public void showNewDatabasePasswdDialog() {
		DialogFragment newDialogFragment =
				PasswdDialogFragment.newInstance(PasswdDialogFragment.passwdDialogType.NEW_DATABASE_PASSWD_DIALOG,
						"");
		newDialogFragment.show(getFragmentManager(), "newDatabasedDialogFragment");
	}

	public static class OldVersionDialogFragment extends DialogFragment {

		public static OldVersionDialogFragment newInstance() {
			return new OldVersionDialogFragment();
		}

		@Override
		public Dialog onCreateDialog(Bundle savedInstanceState) {
			return new AlertDialog.Builder(getActivity())
					.setTitle(R.string.file_old_version_title)
					.setMessage(R.string.file_old_version_msg)
					.setPositiveButton(android.R.string.ok,
							new DialogInterface.OnClickListener() {
								public void onClick(DialogInterface dialog, int whichButton) {
									((Keyring)getActivity()).showPasswdDialog();
								}
							}
					)
					.create();
		}
	}


	public void showOldVersionDialog() {
		DialogFragment newDialogFragment = OldVersionDialogFragment.newInstance();
		newDialogFragment.show(getFragmentManager(), "oldVersionDialogFragment");
	}


	private boolean loadDatabase(String dbFilename, Uri uri, boolean saveFilename) {
		if (dbFilename == null && uri == null) return false;
		if (dbFilename != null && dbFilename.length() == 0) return false;

		if (dbFilename != null) {
			String state = Environment.getExternalStorageState();
			if (!Environment.MEDIA_MOUNTED.equals(state) &&
					!Environment.MEDIA_MOUNTED_READ_ONLY.equals(state)) {
				alertMsg("Media is unavailable");
				return false;
			}
		}

		model = new Model();

		InputStream inStream = null;
		try {
			if (dbFilename != null) {
				//load file from file path
				File myFile = new File(dbFilename);
				inStream = new FileInputStream(myFile);
				model.loadData(inStream);
				mFilename = dbFilename;
				mContentUri = null;
			} else {
				//load file from content uri
				inStream = getContentResolver().openInputStream(uri);
				model.loadData(inStream);
				mContentUri = uri;
				mFilename = null;
			}
		} catch (Exception ex) {
			alertMsg("Failed to open database file.");
			gotoStateIdle();
			if (inStream != null) {
				try {
					inStream.close();
				} catch (Exception ex2) {
					// ignore exceptions
				}
			}
			return false;
		}

		// Don't save file from an incoming intent
		if (saveFilename)
			saveFilename();

		mState = STATE_FILE_LOADED;
		mBodyText.setText("");

		if (model.getVersion() == 4) {
			showOldVersionDialog();
		} else {
			showPasswdDialog();
		}

		return true;
	}


	// Method to obtain file size from a content uri. No longer in use.
	@SuppressWarnings({"unused", "RedundantSuppression"})
	private long getUriFileSize(Uri fileUri) {
		if (fileUri == null) return 0;
		Cursor uriCursor = getContentResolver().
				query(fileUri, null, null, null, null);
		int sizeIndex = uriCursor.getColumnIndex(OpenableColumns.SIZE);
		uriCursor.moveToFirst();
		long uriFileSize = uriCursor.getLong(sizeIndex);
		uriCursor.close();
		return uriFileSize;
	}


	private void gotoStateIdle() {
		model = null;
		mFilename = null;
		mContentUri = null;
    	mState = STATE_IDLE;
		mBodyText.setText(R.string.no_file_loaded_text);
	}

	
	private void saveFilename() {
		String fileStr;
		if (mFilename != null && mContentUri == null)
			fileStr = mFilename;
		else if (mFilename == null && mContentUri != null)
			fileStr = mContentUri.toString();
		else
			return;

		SharedPreferences settings = getPreferences(0);
		String settingsStr = settings.getString(KEY_SETTINGS_FILENAME, null);
		if (settingsStr != null && fileStr.compareTo(settingsStr) == 0) return;

		SharedPreferences.Editor editor = settings.edit();
		editor.putString(KEY_SETTINGS_FILENAME, fileStr);
		editor.apply();
	}

	//remove storage dir path from filename
	public String extractFilePath(String filename) {
		if (filename == null) return "";
		int index = filename.indexOf("Android/data/");
		if (index > -1) {
			return "../" + filename.substring(index);
		}
		return filename;
	}

	// the display name of a content uri is usually it's filename
	private String getUriDisplayName(Uri uri) {
		String displayName = "";
		try {
			Cursor cursor = getContentResolver().query(uri, null, null,
					null, null);
			if (cursor != null && cursor.moveToFirst()) {
				int nameIndex = cursor.getColumnIndex(OpenableColumns.DISPLAY_NAME);
				if (!cursor.isNull(nameIndex)) {
					displayName = cursor.getString(nameIndex);
				}
			}
			if (cursor != null) cursor.close();
		}
		// a query for a deleted file will issue a security exception
		catch (SecurityException e) { displayName = ""; }
		return displayName;
	}

	private String getFileDisplayName() {
		String fileStr = "";
		if (mFilename != null) fileStr = extractFilePath(mFilename);
		else if (mContentUri != null) fileStr = getUriDisplayName(mContentUri);
		return fileStr;
	}

	private String getFileShortDisplayName() {
		String fileStr = getFileDisplayName();
		if (mFilename != null) {
			int index = fileStr.lastIndexOf('/');
			if (index > -1 && index + 1 < fileStr.length())
				fileStr = fileStr.substring(index + 1);
		}
		return fileStr;
	}

	private void openedFileStatusMsg() {
		String fileStr = getFileShortDisplayName();
		if (fileStr.length() > 0) {
			String outStr = (mContentUri != null) ? "Opened file from server" : "Opened file";
			outStr += ": \"" + fileStr + "\"";
			Toast.makeText(Keyring.this, outStr, Toast.LENGTH_LONG).show();
		}
	}

	private void savedFileStatusMsg() {
		String fileStr = getFileShortDisplayName();
		if (fileStr.length() > 0) {
			String outStr = (mContentUri != null) ? "Saved file to server" : "Saved file";
			outStr += ": \"" + fileStr + "\"";
			Toast.makeText(Keyring.this, outStr, Toast.LENGTH_LONG).show();
		}
	}
	
	// password -------------------------------------------------------
	/**
	 * Show password dialog and set Keyring database password.
	 */
	private void checkPassword(CharSequence password) {

    	char[] passwd = new char[password.length()];
    	for(int i=0; i < password.length(); i++) {
    		passwd[i] = password.charAt(i);
		}

		try {
			model.crypto.setPassword(passwd);
        }
		catch(Exception ex) {
			badPasswdMsg(ex.getMessage());
			return;
		}

    	//set lock timer
		resetTimeout(); 
		mLockButton.setText(R.string.button_lock);

		if (mState == STATE_FILE_LOADED)
			openedFileStatusMsg();

    	if (mState != STATE_LOCKED)
    		sortEntries(-1);

		mState = STATE_READY;
		if (model.getEntriesSize() == 1) {
			mCurrentEntry = 0;
		}
		if (mCurrentEntry == -1) {
			listAllEntries(null);
		} else {
			showEntry(mCurrentEntry);
		}
	}
	

    private void badPasswdMsg(String msg) {
    	new AlertDialog.Builder(this)
    		.setMessage(msg)
    		.setPositiveButton("OK", 
    			new DialogInterface.OnClickListener() {
    				@Override
    				public void onClick(DialogInterface dialog, int which) {
    					//password failed, try again
						showPasswdDialog();
    				}
    			}).show();
    }
    

	private int sortEntries(int matchId) {
		//sort entries according to title
		Collections.sort(model.getEntries());

		//build list of titles
		mTitleList.clear();
		mTitleIdList.clear();		
    	Entry entry;
    	int returnId = -1;
    	int cnt = 0;
    	for(Enumeration<Entry> e = model.getElements(); e.hasMoreElements(); ) {
			entry = e.nextElement();
	        // add a reference to the list
			mTitleList.add(entry.getTitle());
	        mTitleIdList.add(entry.getEntryId());
	        if (entry.getEntryId() == matchId) returnId = cnt;
	        ++cnt;
		}

    	if (mTitleIdList.size() == 0) {
    		alertMsg("Database contains no entries");
    		gotoStateIdle();
    		return -1;
    	}
		return returnId;
	}
	
	
	private void showEntry(int id) {
		if (mState != STATE_READY) {
			return;
		}

		mBodyText.scrollTo(0,0);
        mCurrentEntry = id;
        Entry card = model.getEntries().get(id);
		mBodyText.setText(String.format(getString(R.string.card_text),
				card.getTitle(), card.getAccount(), card.getPassword(), card.getNotes()));
    	
    	//restart lock timer
		resetTimeout(); 
	}
	

	//set lock timer
	private void resetTimeout() {
	    mHandler.removeCallbacks(mTimeoutCallback);
	    mHandler.postDelayed(mTimeoutCallback, mTimeoutValue);
    	mStartLastTimeout = System.currentTimeMillis();
	}
    

	private void newEntry() {
		if (mReadOnly) {
			readOnlyAlert();
			return;
		}
		if (mState != STATE_READY) {
			issueBadStatusAlert();
			return;
		}
		float textSize = mBodyText.getTextSize();
		Intent i = new Intent(this, EditEntry.class);
        i.putExtra(KEY_ENTRY_ID, -1);
        i.putExtra(KEY_ENTRY_TITLE_TEXT, "");
        i.putExtra(KEY_ENTRY_ACCOUNT_TEXT, "");
        i.putExtra(KEY_ENTRY_PASSWD_TEXT, "");
        i.putExtra(KEY_ENTRY_NOTES_TEXT, "");
        i.putExtra(KEY_ENTRY_TEXT_SIZE, textSize);
        i.putExtra(KEY_DISPLAY_PAGE_TITLE, getString(R.string.new_entry_page_title));
        startActivityForResult(i, ACTIVITY_NEW_ENTRY);
	}
	
	
	private void editEntry() {
		if (mReadOnly) {
			readOnlyAlert();
			return;
		}
		if (mState != STATE_READY) {
			issueBadStatusAlert();
			return;
		}
		if (mCurrentEntry < 0 || mCurrentEntry >= model.getEntriesSize()) return;

        Entry entry = model.getEntries().get(mCurrentEntry);

		float textSize = mBodyText.getTextSize();
		Intent i = new Intent(this, EditEntry.class);
        i.putExtra(KEY_ENTRY_ID, mCurrentEntry);
        i.putExtra(KEY_ENTRY_TITLE_TEXT, entry.getTitle());
        i.putExtra(KEY_ENTRY_ACCOUNT_TEXT, entry.getAccount());
        i.putExtra(KEY_ENTRY_PASSWD_TEXT, entry.getPassword());
        i.putExtra(KEY_ENTRY_NOTES_TEXT, entry.getNotes());
        i.putExtra(KEY_ENTRY_TEXT_SIZE, textSize);
        i.putExtra(KEY_DISPLAY_PAGE_TITLE, getString(R.string.new_entry_page_title));
        startActivityForResult(i, ACTIVITY_EDIT_ENTRY);
	}
	
	
	private void deleteEntryDialog() {
		if (mReadOnly) {
			readOnlyAlert();
			return;
		}
		if (mState != STATE_READY) {
			issueBadStatusAlert();
			return;
		}
		if (model.getEntriesSize() < 2) {
			alertMsg("Cannot delete the last entry. Database with no entries is invalid. " +
					"Add a new entry then use delete.");
			return;
		}
		if (mCurrentEntry < 0 || mCurrentEntry >= model.getEntriesSize()) {
			alertMsg("deleteEntryDialog: bad current entry id");
			return;
		}
		new AlertDialog.Builder(this)
		.setMessage("Are you sure you want to delete this entry?")
		.setPositiveButton("Delete", new DialogInterface.OnClickListener() {
				@Override
				public void onClick(DialogInterface dialog, int which) {
					//yes, delete the card
					deleteEntry(mCurrentEntry);
				}
			})
		.setNegativeButton("Cancel", new DialogInterface.OnClickListener() {
					@Override
					public void onClick(DialogInterface dialog, int which) { }
				})
		.show();
	}
	
	
	private void readOnlyAlert() {
		alertMsg("Keyring is in Read Only Mode. See Menu/Preferences to enable Read/Write Mode.");
	}


	private void issueBadStatusAlert() {
		if (mState == STATE_IDLE) {
			alertMsg("No keyring file loaded. Keyring for Android will only modify an existing keyring file." +
					"To create a new keyring file use KeyringEditor: www.pnewman.com/keyring.");
		}
		else if (mState == STATE_LOCKED) {
			alertMsg("Keyring file is locked. You must first unlock the keyring file before making any changes.");
		}
	}


	private void deleteEntry(int entryId) {
		Entry currentEntry = model.getEntries().get(entryId);
        if (currentEntry.entryId != mTitleIdList.get(entryId)) {
        	alertMsg("Delete entry: bad entry id");
        	return;
        }
        
        model.removeEntry(currentEntry);
		mCurrentEntry = -1;
		sortEntries(-1);

		// save changes
		saveDatabase();

		listAllEntries(null);
	}
	

	//list button: list first lines of all entries
    public void listAllEntries(View view) {   	
    	if (mState == STATE_LOCKED) {
			showPasswdDialog();
    	}   	
    	else if (mState == STATE_READY) {
	    	Intent i = new Intent(this, SearchList.class);
	        i.putStringArrayListExtra(KEY_DISPLAY_LIST, mTitleList);
	        i.putExtra(KEY_DISPLAY_PAGE_TITLE, getString(R.string.list_entries));
	        startActivityForResult(i, ACTIVITY_DISPLAY_LIST);
    	}
    	else if (mState == STATE_IDLE || mState == STATE_FILE_LOADED) {
    		gotoStateIdle();
    	}
    }
    
    
    //lock button: lock database so further use requires password
    public void lockDatabase(View view) {
    	if (mState == STATE_READY) {
        	mState = STATE_LOCKED;
    		mBodyText.setText(R.string.msg_database_locked);
    		mLockButton.setText(R.string.button_unlock);
    	}
    	else if (mState == STATE_LOCKED) {
			showPasswdDialog();
    	}
    	else if (mState == STATE_IDLE || mState == STATE_FILE_LOADED) {
    		gotoStateIdle();
    	}
        mHandler.removeCallbacks(mTimeoutCallback);
    }

    //initiate open file activity
    private void openFileDialog() {
    	String state = Environment.getExternalStorageState();
    	if (!Environment.MEDIA_MOUNTED.equals(state) &&
    		!Environment.MEDIA_MOUNTED_READ_ONLY.equals(state)) {
    			alertMsg("Media is unavailable");
        		return;
    	}
        Intent i = new Intent(this, FileDialog.class);
        startActivityForResult(i, ACTIVITY_GET_FILENAME);
    }

	private void openSafFile() {
		Intent intent = new Intent(Intent.ACTION_OPEN_DOCUMENT);
		intent.addCategory(Intent.CATEGORY_OPENABLE);
		intent.setType("*/*");

		startActivityForResult(intent, ACTIVITY_OPEN_SAF_FILE);
	}

	private void createNewSafFileDialog() {
		new AlertDialog.Builder(this)
				.setMessage(getString(R.string.new_saf_file_query))
				.setPositiveButton("Yes", new DialogInterface.OnClickListener() {
					@Override
					public void onClick(DialogInterface dialog, int which) {
						//yes, create a new database file
						createNewSafFile();
					}
				})
				.setNegativeButton("Cancel", new DialogInterface.OnClickListener() {
					@Override
					public void onClick(DialogInterface dialog, int which) {
					}
				})
				.show();
	}

	private void createNewSafFile() {
		Intent intent = new Intent(Intent.ACTION_CREATE_DOCUMENT);
		intent.addCategory(Intent.CATEGORY_OPENABLE);
		intent.setType("application/octet-stream");

		startActivityForResult(intent, ACTIVITY_CREATE_NEW_SAF_FILE);
	}

	private Uri getUriFromIntent(Intent intent) {
		if (intent != null) {
			Uri uri = intent.getData();
			if (uri != null) {
				String scheme = uri.getScheme();
				if (scheme != null && scheme.equals("content"))
					return uri;
			}
		}
		return null;
	}

	// preserve access to files from content uri across device restarts
	private void takePersistableUriPermission(Uri uri, Intent intent) {
		final int takeFlags = intent.getFlags()
				& (Intent.FLAG_GRANT_READ_URI_PERMISSION | Intent.FLAG_GRANT_WRITE_URI_PERMISSION);
		getContentResolver().takePersistableUriPermission(uri, takeFlags);
	}

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent intent) {
        super.onActivityResult(requestCode, resultCode, intent);
		//result of create new local saf file
		if (requestCode == ACTIVITY_CREATE_NEW_SAF_FILE && resultCode == RESULT_OK) {
			Uri uri = getUriFromIntent(intent);
			if (uri != null) {
				mTempContentUri = uri;  // temp because Android dialogs are not modal
				takePersistableUriPermission(uri, intent);
				// get new passwd
				showNewDatabasePasswdDialog();
			}
		}
        //result of open local saf file
		if (requestCode == ACTIVITY_OPEN_SAF_FILE && resultCode == RESULT_OK) {
			Uri uri = getUriFromIntent(intent);
			if (uri != null) {
				mCurrentEntry = -1;
				if (loadDatabase(null, uri, true)) {
					takePersistableUriPermission(uri, intent);
				}
			}
		}
        //result of legacy open file dialog
        if (requestCode == ACTIVITY_GET_FILENAME && resultCode == RESULT_OK) {
        	String filename = intent.getStringExtra(KEY_FILE_NAME);
        	if (filename != null && filename.length() > 0) {
        		mCurrentEntry = -1;
        		loadDatabase(filename, null, true);
        	}
        }
        //result of list first lines
        if (requestCode == ACTIVITY_DISPLAY_LIST && resultCode == RESULT_OK) {
        	int result = intent.getIntExtra(KEY_LIST_SELECTION, -1);
        	if (result > -1) {
        		showEntry(result);
        	}
        }
        //result of new entry
        if (requestCode == ACTIVITY_NEW_ENTRY && resultCode == RESULT_OK) {
        	String title = intent.getStringExtra(Keyring.KEY_ENTRY_TITLE_TEXT);
        	byte[] record = createRecord(intent);
        	if (title != null && title.length() > 0 && record.length > 0) addNewEntry(title, record);
        }
        //result of edit entry
        if (requestCode == ACTIVITY_EDIT_ENTRY && resultCode == RESULT_OK) {
        	String title = intent.getStringExtra(Keyring.KEY_ENTRY_TITLE_TEXT);
        	byte[] record = createRecord(intent);
        	if (intent.getIntExtra(Keyring.KEY_ENTRY_ID, -1) == mCurrentEntry &&
        		mCurrentEntry > -1 && title != null && title.length() > 0 && record.length > 0) {
        			editEntry(title, record);
        	}
        }
        //result of set preferences
        if (requestCode == ACTIVITY_SET_PREFERENCES) {
        	restorePreferences();
        }
    }
	

    private byte [] createRecord(Intent intent) {
    	byte[] record = null;
		String account = intent.getStringExtra(Keyring.KEY_ENTRY_ACCOUNT_TEXT);
		String password = intent.getStringExtra(Keyring.KEY_ENTRY_PASSWD_TEXT);
		String notes = intent.getStringExtra(Keyring.KEY_ENTRY_NOTES_TEXT);
    	// set record format & IV length
		switch(model.pdbVersion) {
			case 4:
				record = Model.toRecordFormat4(
						(account != null ? account : "") + "\0" +
								(password != null ? password : "") + "\0" +
								(notes != null ? notes : "") + "\0");
				break;
			case 5:
				record = Model.toRecordFormat5(
						(account != null ? account : ""),
						(password != null ? password : ""),
						(notes != null ? notes : ""));
				break;
		}
		return record;
    }
    
    
	private void addNewEntry(String title, byte[] record) {
		byte[] ciphertext;
		int recordLength = 0;
		int newEntryId = model.getEntriesSize() + 1;
		
		try {
			// encrypt record
			ciphertext = model.crypto.encrypt(record);

			int ivLen = (model.pdbVersion == 5 && model.crypto.type != 1) ? 16 : 8;
			byte[] encodedTitle = Model.stringToByteArray(title);
			int len = encodedTitle.length + ciphertext.length - 16;
			switch (model.pdbVersion) {
				case 4: recordLength = len + 1;	break;
				case 5:	recordLength = len + 4 + (len % 2) + ivLen;	break;
			}

			int maxTextLen = Model.MAX_ENTRY_SIZE - 1;
			if (recordLength >= maxTextLen) {
				alertMsg("New entry: entry length exceeds maximum length for an entry of " + maxTextLen + " bytes.");
				return;
			}

			// get new unique id
			int id = model.getNewUniqueId();

			// new entry object
			Entry myEntry = new Entry(
				newEntryId,
				title,
				0,
				Model.sliceBytes(ciphertext, 16, ciphertext.length - 16),
				model.crypto,
				0x40,
				id,
				recordLength,
				Model.sliceBytes(ciphertext, 0, ivLen));

			// register new entry to vector entries
			model.addEntry(myEntry);

			// save database
			saveDatabase();

			//rebuild list of titles
			mCurrentEntry = sortEntries(newEntryId);
			showEntry(mCurrentEntry);
		}
		catch(Exception ex) {
			msgError(ex, "Add New Entry: ", true);
		}
    }
	

	private void editEntry(String title, byte[] record) {
		try {
			// encrypt record
			byte[] ciphertext = model.crypto.encrypt(record);

			int ivLen = (model.pdbVersion == 5 && model.crypto.type != 1) ? 16 : 8;
			byte[] encodedTitle = Model.stringToByteArray(title);
			int len = encodedTitle.length + ciphertext.length - 16;
			int recordLength = 0;
			switch (model.pdbVersion) {
				case 4: recordLength = len + 1;	break;
				case 5:	recordLength = len + 4 + (len % 2) + ivLen;	break;
			}

	        Entry myEntry = model.getEntries().get(mCurrentEntry);
	        if (myEntry.entryId != mTitleIdList.get(mCurrentEntry)) {
	        	alertMsg("Edit entry: bad entry id");
	        	return;
	        }
			int maxTextLen = Model.MAX_ENTRY_SIZE - 1;
			if (recordLength >= maxTextLen) {
				alertMsg("Edit entry: entry length exceeds maximum length for an entry of " + maxTextLen + " bytes.");
				return;
			}

			myEntry.title = title;
			myEntry.encrypted = Model.sliceBytes(ciphertext, 16, ciphertext.length - 16);
			myEntry.category = 0;
			myEntry.attribute = 0x40;
			myEntry.recordLength = recordLength;
			myEntry.iv = Model.sliceBytes(ciphertext, 0, ivLen);
	        
			// save database
			saveDatabase();

			//rebuild list of titles
			mCurrentEntry = sortEntries(myEntry.entryId);
			showEntry(mCurrentEntry);
		}
		catch(Exception ex) {
			msgError(ex, "Edit Entry: ", true);
		}
    }


	private void saveDatabase() {
		try {
			model.checkRecordLengths();
			OutputStream outStream;
			if (mContentUri != null) {
				// need to explicitly truncate the file for api 29 by opening with mode 'wt'
				outStream = getContentResolver().openOutputStream(mContentUri, "wt");
				model.saveData(outStream);
			}
			else if (mFilename != null) {
				File myFile = new File(mFilename);
				outStream = new FileOutputStream(myFile);
				model.saveData(outStream);
			}
			savedFileStatusMsg();
		}
		catch(Exception ex) {
			msgError(ex, "Save data operation failed.", false);
		}
	}


	private void displayAbout() {
		String versionName = " version ";
		try {
			versionName += getPackageManager().getPackageInfo(getPackageName(), 0).versionName;
		} catch (PackageManager.NameNotFoundException e) {
			versionName = "";
		}
		String aboutInfo = "Keyring for Android" + versionName;
		aboutInfo += "\nCopyright 2023 Peter Newman\n";
		aboutInfo += "www.pnewman.com/keyring\n";
		aboutInfo += "Bug reports to <keyring@pnewman.com>\n\n";

		if (mFilename != null) {
			String fileStr = extractFilePath(mFilename);
			int index = fileStr.lastIndexOf('/');
			if (index > -1 && index + 1 < fileStr.length()) {
				aboutInfo += "File: " + fileStr.substring(index + 1);
				aboutInfo += "\nLocation: " + fileStr.substring(0, index + 1);
			} else {
				aboutInfo += "File: " + fileStr;
			}
		}
		else if (mContentUri != null) {
			aboutInfo += "File: " + getUriDisplayName(mContentUri) + "\nFrom server: " + mContentUri.getAuthority();
		} else {
			aboutInfo += "No file loaded";
		}

		if (model != null) {
			if (model.getVersion() == 5) {
				aboutInfo += "\nFile format version: 5\nAlgorithm: ";
				switch (model.crypto.type) {
					case 1:
						aboutInfo += "Triple DES";
						break;
					case 2:
						aboutInfo += "AES 128-bit";
						break;
					case 3:
						aboutInfo += "AES 256-bit";
						break;
					default:
						aboutInfo += "invalid type";
				}
				aboutInfo += "\nIterations: " + model.getIterations();
			} else if (model.getVersion() == 4) {
				aboutInfo += "\nFile format version: 4\nAlgorithm: Triple DES";
			} else {
				aboutInfo += "\nFile format: invalid version id";
			}
			aboutInfo += "\nEntries: " + model.getEntriesSize();
		}

		aboutInfo +=
			"\n\nKeyring for Android is based on KeyringEditor Copyright 2004 Markus Griessnig.\n\n" +
			"KeyringEditor is based on Java Keyring v0.6 Copyright 2004 Frank Taylor.\n\n" +
			"Both are derived from Keyring for Palm OS: gnukeyring.sourceforge.net.\n\n";

		aboutInfo +=
		"This program is free software; you can redistribute it and/or modify it under " +
		"the terms of the GNU General Public License as published by the Free Software " +
		"Foundation; either version 3 of the License, or (at your option) any later version.\n\n";

		aboutInfo +=
		"This program is distributed in the hope that it will be useful, but WITHOUT ANY " +
		"WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A " +
		"PARTICULAR PURPOSE.  See the GNU General Public License for more details: " +
		"www.gnu.org/licenses/.";
			
		Intent i = new Intent(this, Help.class);
        i.putExtra(KEY_DISPLAY_HELP_FILE, aboutInfo);
        i.putExtra(KEY_DISPLAY_PAGE_TITLE, getString(R.string.about_page_title));
        startActivity(i);
	}
	
	
    private void loadHelpFile() {
    	final int BUFSIZE = 1024;

    	char[] buf = new char[BUFSIZE];
    	InputStream is = getResources().openRawResource(R.raw.helpfile);
    	InputStreamReader isr = new InputStreamReader(is);
    	StringBuilder stringBuf = new StringBuilder();
		int bytesRead = 0;
    	try {
			while (bytesRead > -1) {
    			bytesRead = isr.read(buf, 0, BUFSIZE);
		    	for (int i=0; i < bytesRead; ++i) {
		    		if (buf[i] != '\r') stringBuf.append(buf[i]);
		    	}
			}
			isr.close();
    	}
    	catch (IOException e) {
    		alertMsg("File Read Exception reading help file");
    		return;
    	}

    	String helpFile = stringBuf.toString();
    	if (helpFile.length() == 0) {
    		alertMsg("Problem reading help file");
    		return;
    	}
    	
    	Intent i = new Intent(this, Help.class);
        i.putExtra(KEY_DISPLAY_HELP_FILE, stringBuf.toString());
        i.putExtra(KEY_DISPLAY_PAGE_TITLE, getString(R.string.help_dialog_title));
        startActivity(i);
    }

    private void createNewDatabase(CharSequence password) {
		if (mTempContentUri == null)
			return;

		gotoStateIdle();
		mContentUri = mTempContentUri;
		mTempContentUri = null;
		model = new Model();
		model.initNewDatabase();

		char[] passwd = new char[password.length()];
		for(int i=0; i < password.length(); i++) {
			passwd[i] = password.charAt(i);
		}

		try {
			model.crypto.setNewPassword(passwd);
		}
		catch(Exception ex) {
			gotoStateIdle();
			return;
		}

		String exampleNotes = "Example Notes:\n\nTo add a new entry use Edit/New Entry." +
				"\nTo delete this example entry use Edit/Delete Entry after creating a new entry." +
				"\nFor further information use Help." +
				"\n\nhttp://pnewman.com/keyring/";

		byte[] record = Model.toRecordFormat5("myAccount", "myPasswd", exampleNotes);
		addNewEntry("Example Entry", record);

		mState = STATE_READY;
		mLockButton.setText(R.string.button_lock);
		showEntry(0);
		saveFilename();
	}
    
	//test Password-Based Key Derivation algorithm pbkdf2
	@SuppressWarnings({"unused", "RedundantSuppression"})
	private void testPbkdf2() {
		String passwdStr = "password";
		String saltStr = "ATHENA.MIT.EDUraeburn";
		int iter = 1200;
		int keylen = 32;
		mBodyText.setText(model.testPbkdf2(passwdStr, saltStr, iter, keylen));
	}

	@SuppressWarnings({"unused", "RedundantSuppression"})
	public void showStatusMsgShort(String msg) {
    	Toast.makeText(Keyring.this, msg, Toast.LENGTH_SHORT).show();
    }

	@SuppressWarnings({"unused", "RedundantSuppression"})
	public void showStatusMsgLong(String msg) {
    	Toast.makeText(Keyring.this, msg, Toast.LENGTH_LONG).show();
    }
	
	
    private void msgError(Exception ex, String msg,
						  @SuppressWarnings({"unused", "RedundantSuppression"}) boolean ignore) {
    	alertMsg(msg + " " + ex.getMessage());
    }

    
    private void alertMsg(String msg) {
    	new AlertDialog.Builder(this)
    		.setMessage(msg)
    		.setPositiveButton("OK", 
    			new DialogInterface.OnClickListener() {
    				@Override
    				public void onClick(DialogInterface dialog, int which) { }
    			}).show();
    }
}
