/*
Palm Keyring for Android

Copyright (C) 2022 Peter Newman <pn@pnewman.com>

Palm Keyring for Android is based on:
KeyringEditor v1.1
Copyright 2004 Markus Griessnig
Vienna University of Technology Institute of Computer Technology

KeyringEditor is based on:
Java Keyring v0.6
Copyright 2004 Frank Taylor <keyring@lieder.me.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE.  See the GNU General Public License for more details:
<http://www.gnu.org/licenses/>.
*/

// 06.08.2022: fix file browser can select incorrect file if filename starts with non-alphanumeric char
// 06.08.2022: display "Folder is empty" text if directory is empty

package com.pnewman.keyring;

import android.app.AlertDialog;
import android.app.ListActivity;
import android.content.DialogInterface;
import android.content.Intent;
import android.os.Bundle;
import android.view.View;
import android.widget.ArrayAdapter;
import android.widget.ListView;
import android.widget.TextView;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

public class FileDialog extends ListActivity {
    private static final String root = "/";

	private static class Entry {
		String item;	// name string presented to user
		String path;	// full path of file

		public Entry(String itemStr, String pathStr) {
			item = itemStr;
			path = pathStr;
		}
	}

	private List<Entry> entries = null;

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
        setContentView(R.layout.file_dialog);
        setTitle(R.string.file_dialog_title);

		// (Environment.getExternalStorageDirectory() not supported after version 29)
		File extDir = this.getExternalFilesDir(null);
		if (extDir != null)
			getDir(extDir.getAbsolutePath());
		else
			alertMsg("Failed to open external files folder.");
	}

    private void getDir(String dirPath) {
		TextView myPath = findViewById(R.id.path);
		myPath.setText(String.format(getString(R.string.file_dialog_path), dirPath));
		TextView footer = findViewById(R.id.dir_empty);
		footer.setText("");

        entries = new ArrayList<>();
        
        File f = new File(dirPath);
        File[] files = f.listFiles(new FileFilter() {
			@Override
			public boolean accept(File pathname) {
				//If a file or directory is hidden, or unreadable, don't show it in the list.
				if (pathname.isHidden())
					return false;
				return pathname.canRead();
			}
		});

        if (!dirPath.equals(root)) {
			entries.add(new Entry("../", f.getParent()));
        }

        int numFiles = 0;
		if (files != null) {
			for (File file : files) {
				if (file.isDirectory())
					entries.add(new Entry(file.getName() + "/", file.getPath()));
				else
					entries.add(new Entry(file.getName(), file.getPath()));
				++numFiles;
			}
		}

		if (numFiles == 0)
			footer.setText(R.string.empty_folder);

        // Sort the files alphabetically.
		// (Sort on full pathname to avoid problem of files starting with chars before '.')
		Collections.sort(entries, new Comparator<Entry>() {
			@Override
			public int compare(Entry entry01, Entry entry02) {
				return entry01.path.compareToIgnoreCase(entry02.path);
			}
		});

		List<String> items = new ArrayList<>();
		for (Entry entry : entries)
			items.add(entry.item);

        ArrayAdapter<String> fileList =
				new ArrayAdapter<>(this, R.layout.row, items);
        setListAdapter(fileList);
    }
    
    @Override
    protected void onListItemClick(ListView l, View v, int position, long id) {
     
		File file = new File(entries.get(position).path);
     
    	if (file.isDirectory()) {
    		if (file.canRead()) {
				getDir(entries.get(position).path);
    		} else {
				alertMsg("Access denied: " + file.getName());
    		}
    	} else {
	    	Intent mIntent = new Intent();
	    	mIntent.putExtra(Keyring.KEY_FILE_NAME, file.getAbsolutePath());
	    	setResult(RESULT_OK, mIntent);
	    	finish();
    	}
    }

	private void alertMsg(String msg) {
		new AlertDialog.Builder(this)
			.setMessage(msg)
			.setPositiveButton("OK",
				new DialogInterface.OnClickListener() {
					@Override
					public void onClick(DialogInterface dialog, int which) { }
				}).show();
	}
}
