/*
KeyringEditor

Copyright 2004 Markus Griessnig
Vienna University of Technology
Institute of Computer Technology

KeyringEditor is based on:
Java Keyring v0.6
Copyright 2004 Frank Taylor <keyring@lieder.me.uk>

These programs are distributed in the hope that they will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.
*/

// Prop.java

// 24.11.2004

// 07.12.2004: csvFilename deleted
// 16.01.2022: add limits on timeout value
// 23.01.2022: add full path to init file

package com.pnewman.apps.keyring;

import java.util.*;
import java.io.*;
import static java.io.File.separator;

/**
 * This class is used to load parameters from the file keyringeditor.ini.
 */
public class Prop {
    // ----------------------------------------------------------------
    // variables
    // ----------------------------------------------------------------

    /**
     * Default filename
     */
    private static final String INI_FILENAME = "keyringeditor.ini";

    /**
     * Reference to class Editor
     */
    private final KeyringEditor editor;

    // ----------------------------------------------------------------
    // constructor
    // ----------------------------------------------------------------
    /**
     * Default constructor.
     *
     * @param editor Reference to class Editor
     */
    public Prop(KeyringEditor editor) {
        this.editor = editor;
    }

    // ----------------------------------------------------------------
    // public ---------------------------------------------------------
    // ----------------------------------------------------------------
    /**
     * This method opens the file keyringeditor.ini and reads the
     * parameters "TitleSeparator", "CsvSeparator" and "PasswordTimeout".
     *
     * If no file is found, default values are used.
     *
     * TitleSeparator separates levels in an entry title for the tree view ('/').
     * CsvSeparator is used as the separator for converting entries to a csv-file (';').
     * PasswordTimeout is the time in minutes after inactivity forces a lock of the application ('1').
     */
    public void setup() {
        Properties props = new Properties();

        // load KeyringEditor.ini
        String path = "";
        try {
            path = new File(this.getClass().getProtectionDomain()
                        .getCodeSource().getLocation().getPath()).getParent();
            if (path != null)
                path += separator + INI_FILENAME;
            else
                path = INI_FILENAME;
            
            try (FileInputStream in = new FileInputStream(path)) {
                props.load(in);
                System.err.println("Prop.java: Opened init file: " + path);
            }
        }
        catch(IOException e) {
            System.err.println("Prop.java: File " + path + " not found. Using default values.");
            return;
        }

        //props.list(System.out); // debug

        /*
        String csvFilename = props.getProperty("CsvFilename");
        if(csvFilename != null) {
                editor.getModel().setCsvFilename(csvFilename); // Default: 'keyring.csv'
        }
        */

        // set properties
        String titleSeparator = props.getProperty("TitleSeparator");
        if(titleSeparator != null) {
            editor.SEPARATOR = titleSeparator.charAt(0);  // Default: '/'
        }

        String csvSeparator = props.getProperty("CsvSeparator");
        if(csvSeparator != null) {
            editor.getModel().setCsvSeparator(csvSeparator.charAt(0));  // Default: ','
        }

        String pwTimeout = props.getProperty("PasswordTimeout");
        if(pwTimeout != null) {
            int timeout = Integer.parseInt(pwTimeout); // minutes
            if (timeout > 0 && timeout < 400)
                KeyringEditor.PASSWORD_TIMEOUT = timeout * 60 * 1000;  // ms // Default: 20 minutes
//            System.out.println("timeout set = " + timeout + " mins");
        }
    }
}
