/*
KeyringEditor 1.2
Copyright 2022 Peter Newman
pnewman.com/keyring

KeyringEditor is based on:
KeyringEditor v1.1
Copyright 2004 Markus Griessnig
Vienna University of Technology
Institute of Computer Technology

KeyringEditor is based on:
Java Keyring v0.6
Copyright 2004 Frank Taylor <keyring@lieder.me.uk>

These programs are distributed in the hope that they will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.
 */

// Editor.java

// 25.11.2004

// 01.12.2004: Keyring database format 5 support added
// 05.12.2004: MenuItem Tools - Convert database added
// 12.01.2004: Model.writeNewDatabase() in main() added
// 24.05.2005: showEntry - check no category
// 13.05.2018: change show passwd default to true
// 13.05.2018: add lockAndClearScreen() to clear screen on timeout
// 12.06.2021: add warning that version 4 file format is no longer secure
// 19.06.2021: prevent save csv file from overwriting database file
// 26.06.2021: redisplay modified entry after save changes
// 27.06.2021: add application icons for Windows and Mac
// 03.07.2021: add quickstart help
// 25.09.2021: add confirm dialog for delete button
// 02.10.2021: add confirm dialog for unsaved changes when another entry selected
// 15.01.2022: add filename and file details to About dialog


package com.pnewman.apps.keyring;

import java.awt.Color;
import java.awt.Desktop;
import java.awt.Image;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.Vector;
import java.util.prefs.Preferences;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JMenuBar;
import javax.swing.JOptionPane;
import javax.swing.JSplitPane;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreePath;

/**
 * This class handles the gui.
 */
public class KeyringEditor extends Gui {
    // ----------------------------------------------------------------
    // variables
    // ----------------------------------------------------------------
    /**
     * Separates levels in an entry title for the tree view
     */
    protected char SEPARATOR = '/'; // entry title separator

    /**
     * Last directory to load from
     */
    private File previousDirectory = null; // last directory to load from

    /**
     * Current loaded keyring database
     */
    private String pdbFilename = "dummy.pdb"; // default database to save to

    private Gui gui;
    private Model model;
    private JFrame frame;
    private PasswordTimeoutWorker timeoutThread;

    // flags
    /**
     * Category-filter (0 = show all)
     */
    private final int filterCategory = -1;

    /**
     * Show button "Save" only when entry text changes (true)
     */
    private boolean textFieldChanged = true; // show button save only when text changes

    /**
     * Show entry password in clear text (true)
     */
    private boolean showPassword = true;

    /**
     * True when application is locked
     */
    private boolean locked = false;
    
    /**
     * True when quickstart guide displayed
     */
    private boolean quickstartDisplay = false;
    
    private static final String APPLE_MAC_EXTENSIONS = "com.apple.eawt.Application";
    
    private static final String PREF_CURRENT_FILENAME = "current_filename";

    // ----------------------------------------------------------------
    // main -----------------------------------------------------------
    // ----------------------------------------------------------------
    /**
     * main
     *
     * @param args command line parameters
     */
    public static void main(String[] args) {
        String dbFilename = null;

        KeyringEditor myEditor = new KeyringEditor();
/*
        try {
            // set system look and feel
            UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
        } 
        catch (UnsupportedLookAndFeelException | ClassNotFoundException |
               InstantiationException | IllegalAccessException e) {
            myEditor.msgError(e, "Failed to set system look and feel.", false);
        }
*/        
        myEditor.frame = new JFrame(FRAMETITLE);
        myEditor.frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

        Preferences prefs =
                Preferences.userNodeForPackage(com.pnewman.apps.keyring.KeyringEditor.class);
        
        // check command line parameters
        if(args.length > 1) {
            System.out.println("Usage: java -jar KeyringEditor.jar [keyring-database]");
            return;
        }

        if(args.length > 0) {
            String filename = args[0];
            if (!filename.isEmpty()) {
                File currentFile = new File(filename);
                if (currentFile.isFile()) {
                    dbFilename = args[0];
                }
            }
        }
        if (dbFilename == null) {
            String filename = prefs.get(PREF_CURRENT_FILENAME, "");
            if (!filename.isEmpty()) {
                File currentFile = new File(filename);
                if (currentFile.isFile()) {
                    dbFilename = filename;
                }
            }
        }
                
        // setup gui
        try {
            myEditor.setupGui(dbFilename);
        }
        catch(Exception e) {
            myEditor.msgError(e, "main", true);
        }

        //System.out.println(System.getProperty("java.version"));
    }

    // ----------------------------------------------------------------
    // public ---------------------------------------------------------
    // ----------------------------------------------------------------

    /**
     * Returns filename of the current loaded keyring database.
     *
     * @return Filename
     */
    public String getFilename() {
        return this.pdbFilename;
    }

    /**
     * Returns separator for title levels.
     *
     * @return Separator
     */
    public char getSeparator() {
        return this.SEPARATOR;
    }

    /**
     * Returns reference to class Model.
     *
     * @return Reference to class Model
     */
    public Model getModel() {
        return this.model;
    }

    /**
     * pn: added so that lock timer can clear screen
     */
    public void lockAndClearScreen() {
        setBtnLock(true, true);
        enableButtons(false);
        clearEntry();
    }

    // ----------------------------------------------------------------
    // private --------------------------------------------------------
    // ----------------------------------------------------------------

    /**
     * Checks if a file already exists and show warning dialog.
     *
     * @param filename Filename
     *
     * @return False if file operation should be cancelled
     */
    private boolean checkFile(String filename) {
        boolean ok = true;

        File f = new File(filename);

        if(f.exists() == true) {
            int n = JOptionPane.showConfirmDialog(
                frame, "File already exists. Continue?",
                "Warning",
                JOptionPane.YES_NO_OPTION);

            if(n == JOptionPane.NO_OPTION) {
                ok = false;
            }
        }

        return ok;
    }

    /**
     * Shows a error message.
     *
     * @param e Exception
     * @param info User-defined text
     * @param showStack True if Exception Stack Trace should be displayed
     */
    private void msgError(Exception e, String info, boolean showStack) {
        JOptionPane.showMessageDialog(frame,
            info + ": " + e.getMessage(),
            "Keyring Error",
            JOptionPane.ERROR_MESSAGE);

        if(showStack) {
            e.printStackTrace(System.err);
        }
    }

    /**
     * Shows an error message without an exception.
     *
     * @param msg User-defined text
     */
    private void msgAlert(String msg) {
        JOptionPane.showMessageDialog(frame,
            msg,
            "Keyring Error",
            JOptionPane.ERROR_MESSAGE);
    }
    
    /**
     * Shows a information message.
     *
     * @param info User-defined text
     */
    private void msgInformation(String info) {
        JOptionPane.showMessageDialog(frame,
            info,
            "Keyring Information",
            JOptionPane.INFORMATION_MESSAGE);
    }

    /**
     * Confirm dialog.
     *
     * @param question Confirm query
     */    
    private int msgConfirm(String query) {
        return JOptionPane.showConfirmDialog(
            frame, query, "Keyring Confirm",
            JOptionPane.YES_NO_OPTION,
            JOptionPane.QUESTION_MESSAGE);
    }


    // setupGui -------------------------------------------------------
    /**
     * Loads menubar, adds ActionListeners, starts PasswordTimeout Thread.
     *
     * @param dbFilename Keyring database or null
     */
    private void setupGui(String dbFilename) throws Exception {
    // Function: setup gui
    // Parameters: keyring-database
    // Returns: -

        // MenuBar
        JMenuBar myMenuBar = setMenuBar();
        frame.setJMenuBar(myMenuBar);

        // Layout
        JSplitPane mySplitPane = setLayout(this);
        frame.setContentPane(mySplitPane);

        // MenuBar Listener
        openMenuItem.addActionListener(new OpenListener(this));
        closeMenuItem.addActionListener(new CloseListener(this));
        quitMenuItem.addActionListener(new QuitListener(this));
        categoriesMenuItem.addActionListener(new editCategoriesListener(this));
        csvMenuItem.addActionListener(new csvListener(this));
        aboutMenuItem.addActionListener(new AboutListener(this));
        quickstartMenuItem.addActionListener(new QuickstartListener(this));
        convertMenuItem.addActionListener(new convertListener(this));
        newMenuItem.addActionListener(new newListener(this));

        // entryPane Listener
        currentCategory.addActionListener(new currentCategorySelectionListener(this));
        currentTitle.getDocument().addDocumentListener(new documentListener(this));
        currentAccountName.getDocument().addDocumentListener(new documentListener(this));
        currentPassword.getDocument().addDocumentListener(new documentListener(this));
        currentNotes.getDocument().addDocumentListener(new documentListener(this));

        // entryListPane Listener
        categoryList.addActionListener(new CategorySelectionListener(this));
        dynTree.getTree().addTreeSelectionListener(new treeSelectionListener(this));

        // buttonPane Listener
        newEntry.addActionListener(new newEntryListener(this));
        saveEntry.addActionListener(new saveEntryListener(this));
        delEntry.addActionListener(new delEntryListener(this));
        btnLock.addActionListener(new PasswordLockListener(this));
        currentPasswordShow.addActionListener(new PasswordShowListener(this));

        // add mouse listener to catch double click to open url in browser
        currentNotes.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent evt) {
                if (evt.getClickCount() == 2) {
                    checkForURL();
                }
            }
        });
        
        // Frame
        frame.pack();
        frame.setVisible(true);
        frame.setLocation(0, 120);

        // add application icons
        ArrayList<Image> imageList = new ArrayList<>();
        imageList.add(createImage("keyring-icon-16.png"));
        imageList.add(createImage("keyring-icon-32a.png"));
        imageList.add(createImage("keyring-icon-96.png"));
        frame.setIconImages(imageList);
        
        // add application dock icon for a mac
        // (we have to do this via reflection because the apple extensions only exist on a mac)
        // (https://stackoverflow.com/questions/13400926/how-can-i-call-an-os-x-specific-method-for-my-cross-platform-jar)
        if (systemIsAMac()) {
            Image macIcon = createImage("keyring-icon-96.png");
            try {
                // Replace: import com.apple.eawt.Application
                Class<?> cls = Class.forName(APPLE_MAC_EXTENSIONS);
                // Replace: Application application = Application.getApplication();
                Object application = cls.newInstance().getClass().getMethod("getApplication").invoke(null);
                // Replace: application.setDockIconImage(image);
                application.getClass().getMethod("setDockIconImage", java.awt.Image.class).invoke(application, macIcon);
            }
            catch (ClassNotFoundException | IllegalAccessException | IllegalArgumentException |
                   InvocationTargetException | NoSuchMethodException | SecurityException |
                   InstantiationException e) {
                /* ignore exceptions */
            }
        }
        
        // Password Timeout
        timeoutThread = new PasswordTimeoutWorker(this);
        new Thread(timeoutThread).start();

        if (dbFilename == null) {
            displayQuickstartFile();
        }
        
        // load Database
        loadDatabase(dbFilename);
    }

    // loadDatabase ---------------------------------------------------
    /**
     * Loads a Keyring database and setup gui (buttons, menubar) properly.
     *
     * @param dbFilename Keyring database or null
     */
    @SuppressWarnings("empty-statement")
    private void loadDatabase(String dbFilename) throws Exception {
    // Function: load data model
    // Parameters: keyring-database
    // Returns: -
        String[] dbType = {"TripleDES", "TripleDES", "AES128", "AES256"};

        if(dbFilename != null) {
            model = new Model();

            try {
                model.loadData(dbFilename);

                pdbFilename = dbFilename;

                setupProperties(model.getElements());
            }
            catch(Exception ex) {
                msgError(ex, "Open keyring database", false);

                try {
                    loadDatabase(null);
                    return;
                }
                catch(Exception ignore) {};
            }

            // Password dialog
            if(checkPassword() == false) {
                loadDatabase(null);
                return;
            }

            if (model.getVersion() == 4) {
                msgInformation(
                    "This is a version 4 file, an old file format that is no longer secure.\n" +
                    "Please use Tools/Convert database to update the format to version 5\n" +
                    "with AES128 or AES256 encryption. For help see pnewman.com/keyring.");
            }
            
            // initialize buttons etc.
            frame.setTitle(FRAMETITLE + ": " + dbFilename + " (" + dbType[model.crypto.type] + " | Database format " + model.pdbVersion + ")");
            setupCategories(this.model.getCategories());
            setMenuBar(false);
            enableButtons(true);
            setBtnLock(false, true);
            dynTree.populate();
            currentNotes.setText("");
            
            // save filename in persistent state
            Preferences prefs =
                Preferences.userNodeForPackage(com.pnewman.apps.keyring.KeyringEditor.class);
            prefs.put(PREF_CURRENT_FILENAME, dbFilename);
        }
        else {
            model = null;

            // initialize buttons etc.
            frame.setTitle(FRAMETITLE);
            setupCategories(null);
            setMenuBar(true);
            enableButtons(false);
            setBtnLock(false, false);
            dynTree.clear();
        }
        quickstartDisplay = false;

        //frame.show();
    }

    /**
     * Set level separator to display correct entry title.
     *
     * @param entries Keyring entries
     */
    private void setupProperties(Enumeration entries) {
        // set properties
        Prop myProp = new Prop(this);
        myProp.setup();

        // set separator in entry objects
        for(Enumeration e = entries; e.hasMoreElements(); ) {
            Entry entry = (Entry)e.nextElement();
            entry.setTitleSeparator(SEPARATOR);
        }
    }

    /**
     * Enable menubar items according to loaded database.
     *
     * @param False if database is loaded
     */
    private void setMenuBar(boolean open) {
        openMenuItem.setEnabled(open);
        closeMenuItem.setEnabled(!open);
        csvMenuItem.setEnabled(!open);
        categoriesMenuItem.setEnabled(!open);
        convertMenuItem.setEnabled(!open);
    }

    /**
     * Enable buttons according to loaded database.
     *
     * @param enabled True if database is loaded
     */
    private void enableButtons(boolean enabled) {
         delEntry.setEnabled(false);
         saveEntry.setEnabled(false);
         newEntry.setEnabled(enabled);

         enableFields(enabled);
    }

    /**
     * Enable entry fields according to loaded database.
     *
     * @param flag True if database is loaded
     */
    private void enableFields(boolean flag) {
        currentCategory.setEditable(false);
        currentTitle.setEditable(flag);
        currentAccountName.setEditable(flag);
        currentPassword.setEditable(flag);
        currentNotes.setEditable(flag);
    }

    /**
     * Enable button lock according to loaded database and status of password timeout.
     *
     * @param locked True if application is locked
     * @param enabled True if button "Lock" should be enabled
     */
    private void setBtnLock(boolean locked, boolean enabled) {
        btnLock.setText(locked ? "Unlock" : "Lock");
        btnLock.setEnabled(enabled);
        this.locked = locked;

        saveEntry.setBackground(null);
    }

    // categories -----------------------------------------------------
    /**
     * Setup category filter combobox.
     *
     * @param myCategories Categories loaded with Keyring database
     */
    //private void setupCategories(Vector<String> myCategories) { // Java 1.5
    private void setupCategories(Vector myCategories) {
        // categoryList
        //Vector<String> displayCategories; // Java 1.5
        Vector displayCategories;
        if(myCategories != null)
            //displayCategories = new Vector<String>(myCategories); // Java 1.5
            displayCategories = (Vector)myCategories.clone();
        else
            //displayCategories = new Vector<String>(); // Java 1.5
            displayCategories = new Vector();

        displayCategories.add(0, "All");
        categoryList.setModel(new DefaultComboBoxModel(displayCategories));

        // currentCategory
        Vector currentCategories;
        if(myCategories != null)
            currentCategories = (Vector)myCategories.clone();
        else
            currentCategories = new Vector();

        currentCategory.setModel(new DefaultComboBoxModel(currentCategories));
    }

    // password -------------------------------------------------------
    /**
     * Show password dialog and set Keyring database password.
     *
     * @return False if dialog cancelled (Boolean)
     */
    private boolean checkPassword() {
        try {
            model.crypto.setPassword(getPasswordDialog());
            timeoutThread.restartTimeout();
            return true;
        }
        catch(CancelledException e) {
            timeoutThread.setTimeout(); // timed out
            return false;
        }
        catch(Exception e) {
            msgError(e, "checkPassword", false);
            timeoutThread.setTimeout(); // timed out
            return false;
        }
    }

    /**
     * Show password dialog.
     *
     * @return Password
     */
    private char[] getPasswordDialog() throws Exception {
        PasswordDialog pwdDlg = new PasswordDialog(frame, pdbFilename);
        pwdDlg.pack();
        pwdDlg.setVisible(true);

        return pwdDlg.getPassword();
    }

    // showEntry ------------------------------------------------------
    /**
     * Show entry and check password timeout.
     */
    @SuppressWarnings("empty-statement")
    private void showEntry() {
        DefaultMutableTreeNode node = dynTree.getLastNode();

        if(locked == true) {
            return;
        }

        try {
            Date ende = timeoutThread.getEndDate();
            if(ende == null) { // timed out
                clearEntry();

                setBtnLock(true, true);
                enableButtons(false);
                saveEntry.setBackground(null);

                return;
            }
            else {
                timeoutThread.restartTimeout();
            }

            // no entry
            if(node == null || !(node.isLeaf()) || node.isRoot()) {
                clearEntry();

                saveEntry.setEnabled(false);
                saveEntry.setBackground(null);
                delEntry.setEnabled(false);

                return;
            }

            Object nodeInfo = node.getUserObject();

            Entry e = (Entry)nodeInfo;

            // set text fields according to entry
            // if categoryname was deleted, show first category "no category"
            if(currentCategory.getItemCount() > e.getCategory()) {
                currentCategory.setSelectedIndex(e.getCategory());
            }
            else {
                currentCategory.setSelectedIndex(0); // no category
            }

            currentTitle.setText(e.getTitle());
            currentAccountName.setText(e.getAccount());
            currentNotes.setText(e.getNotes());
            currentNotes.setCaretPosition(0);
            currentPassword.setText(e.getPassword());
            currentDate.setText(e.getDate());

            // initialize buttons
            textFieldChanged = false;
            saveEntry.setEnabled(false);
            saveEntry.setBackground(null);
            delEntry.setEnabled(true);
            quickstartDisplay = false;
        }
        catch(Exception e) {
            msgError(e, "showEntry", true);

            try {
                loadDatabase(null);
            }
            catch(Exception ignore) {};
        }
    }

    /**
     * Clear entry text fields.
     */
    private void clearEntry() {
        //currentCategory.setSelectedIndex(0);
        currentTitle.setText("");
        currentAccountName.setText("");
        currentPassword.setText("");
        currentNotes.setText("");
        currentDate.setText("");
        quickstartDisplay = false;
    }
    
    // returns an Image, or null if the path was invalid
    public Image createImage(String path) {
        URL imgURL = KeyringEditor.class.getResource("resources/" + path);
        if (imgURL != null) {            
            Toolkit kit = Toolkit.getDefaultToolkit();    
            return kit.createImage(imgURL);
        } else {
            return null;
        }
    }
    
    private boolean systemIsAMac() {
        try {
            Class.forName(APPLE_MAC_EXTENSIONS, false, null);
            return true;
        }
        catch (ClassNotFoundException ex) {
            return false;
        }
    }

    private void displayQuickstartFile() {
        String textFile = "KeyringEditor version " + KeyringEditor.VERSION + "\n\n";
        try (InputStream stream = KeyringEditor.class.getResourceAsStream("resources/quickstart.txt")) {
            BufferedReader r = new BufferedReader(new InputStreamReader(stream, "UTF-8"));

            String line;
            while((line = r.readLine()) != null) {
                textFile += line + "\n";
            }
            
            currentNotes.setText(textFile);
            currentNotes.setCaretPosition(0);
        }
        catch(IOException e) {
            msgError(e, "Display text resource", false);
        }
    }
    
    // ----------------------------------------------------------------
    // listener menubar -----------------------------------------------
    // ----------------------------------------------------------------

    /**
     * MenuItem Open: shows a File dialog and loads a Keyring database.
     */
    public class OpenListener implements ActionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        protected OpenListener(KeyringEditor editor) {
            this.editor = editor;
        }

        /**
         * This method opens a file dialog and loads the chosen database.
         *
         * @param e the ActionEvent to process
         */
        @Override
        @SuppressWarnings("empty-statement")
        public void actionPerformed(ActionEvent e) {
            JFileChooser chooser = new JFileChooser();

            chooser.setDialogTitle("Open Keyring database");
            chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
            chooser.setMultiSelectionEnabled(false);
            chooser.setCurrentDirectory(previousDirectory);
            FileNameExtensionFilter filter = new FileNameExtensionFilter("Keyring '.pdb' file", "pdb");
            chooser.setFileFilter(filter);
            
            int returnVal = chooser.showOpenDialog(editor.frame);

            if(returnVal == JFileChooser.APPROVE_OPTION) {
                try {
                    File selectedFile = chooser.getSelectedFile();
                    previousDirectory = selectedFile.getParentFile();
                    pdbFilename = selectedFile.getCanonicalPath();
                    editor.loadDatabase(pdbFilename);
                }
                catch(Exception ex) {
                    msgError(ex, "Open Keyring database", false);

                    try {
                        editor.loadDatabase(null);
                    }
                    catch(Exception ignore) {};
                }
            }
        }
    }

    /**
     * MenuItem Close: close Keyring database.
     */
    public class CloseListener implements ActionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        protected CloseListener(KeyringEditor editor) {
            this.editor = editor;
        }

        /**
         * This method closes the loaded database.
         *
         * @param e the ActionEvent to process
         */
        @Override
        @SuppressWarnings("empty-statement")
        public void actionPerformed(ActionEvent e) {
            try {
                editor.loadDatabase(null);
            }
            catch(Exception ignore) {};
        }
    }

    /**
     * MenuItem Quit: exit Application.
     */
    public class QuitListener implements ActionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        protected QuitListener(KeyringEditor editor) {
            this.editor = editor;
        }

        /**
         * This method ends the application.
         *
         * @param e the ActionEvent to process
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            System.exit(0);
        }
    }

    /**
     * MenuItem Edit categories: shows the categories dialog for editing category-names.
     */
    public class editCategoriesListener implements ActionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        protected editCategoriesListener(KeyringEditor editor) {
            this.editor = editor;
        }

        /**
         * This method opens the categories dialog and updates the category combo boxes.
         *
         * @param e the ActionEvent to process
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            if(locked == true) {
                msgInformation("Please unlock application first.");
                return;
            }

            // show category dialog
            CategoriesDialog catDialog = new CategoriesDialog(frame, model.getCategories());

            catDialog.pack();
            catDialog.setVisible(true);

            // update category combo boxes
            Vector<String> newCategories = catDialog.getNewCategories();
            if(newCategories != null) { // changed categories
                setupCategories(newCategories);
                model.setCategories(newCategories);

                // save database
                try {
                    editor.model.saveData(pdbFilename);
                }
                catch(Exception ex) {
                    msgError(ex, "Could not save entries to " + pdbFilename, false);
                }
            }
        }
    }

    /**
     * MenuItem Save database to csv: save the loaded Keyring database as a CSV file.
     */
    public class csvListener implements ActionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        protected csvListener(KeyringEditor editor) {
                this.editor = editor;
        }

        /**
         * This method opens a file dialog and saves the database entries to a csv file.
         *
         * @param e the ActionEvent to process
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            if(locked == true) {
                msgInformation("Please unlock application first.");
                return;
            }

            // show File dialog
            JFileChooser chooser = new JFileChooser();

            chooser.setDialogTitle("Save Keyring database to CSV-File");
            chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
            chooser.setMultiSelectionEnabled(false);
            chooser.setCurrentDirectory(previousDirectory);

            int returnVal = chooser.showSaveDialog(editor.frame);

            if(returnVal == JFileChooser.APPROVE_OPTION) {
            try {
                    File selectedFile = chooser.getSelectedFile();
                    previousDirectory = selectedFile.getParentFile();
                    String csvFilename = selectedFile.getCanonicalPath();

                    if (csvFilename.equalsIgnoreCase(pdbFilename)) {
                        msgAlert("Cannot overwrite database file: " + pdbFilename);
                        return;
                    }
                    
                    // check if file exists
                    boolean ok = checkFile(csvFilename);

                    if(ok == true) {
                        // save entries to csv file
                        editor.model.saveEntriesToFile(csvFilename);

                        msgInformation("Entries saved to: " + model.getCsvFilename());
                    }
                }
                catch(Exception ex) {
                    msgError(ex, "Could not save entries to " + model.getCsvFilename(), false);
                }
            }
        }
    }

    /**
     * MenuItem Convert database: shows convert dialog and convert loaded database.
     */
    public class convertListener implements ActionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        protected convertListener(KeyringEditor editor) {
                this.editor = editor;
        }

        /**
         * This method opens the convert dialog and closes the loaded database.
         *
         * @param e the ActionEvent to process
         */
        @Override
        @SuppressWarnings("empty-statement")
        public void actionPerformed(ActionEvent e) {
            if(locked == true) {
                msgInformation("Please unlock application first.");
                return;
            }

            // show convert dialog
            ConvertDialog catDialog = new ConvertDialog(editor.frame, editor.model, editor.model.pdbVersion);

            catDialog.pack();
            catDialog.setVisible(true);

            if(catDialog.getCancelled() == false) {
                msgInformation("Database converted.");

                // close old database
                try {
                    editor.loadDatabase(null);
                }
                catch(Exception ignore) {};
            }
        }
    }

    /**
     * MenuItem New minimal database: generates a new minimal database
     */
    public class newListener implements ActionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        protected newListener(KeyringEditor editor) {
            this.editor = editor;
        }

        /**
         * This method opens a file dialog and generates a new minimal database.
         *
         * @param e the ActionEvent to process
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            // show File dialog
            JFileChooser chooser = new JFileChooser();

            chooser.setDialogTitle("Generate new minimal database");
            chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
            chooser.setMultiSelectionEnabled(false);
            chooser.setCurrentDirectory(previousDirectory);
            FileNameExtensionFilter filter = new FileNameExtensionFilter("Keyring '.pdb' file", "pdb");
            chooser.setFileFilter(filter);
 
            int returnVal = chooser.showSaveDialog(editor.frame);

            if(returnVal == JFileChooser.APPROVE_OPTION) {
                try {
                    File selectedFile = chooser.getSelectedFile();
                    previousDirectory = selectedFile.getParentFile();
                    String newFilename = selectedFile.getCanonicalPath();
                    if(!newFilename.toLowerCase().endsWith(".pdb")) {
                        newFilename += ".pdb";
                    }
                    
                    // check if file exists
                    boolean ok = checkFile(newFilename);

                    if(ok == true) {
                        // generate new minimal database
                        Model.writeNewDatabase(newFilename);

                        msgInformation("New minimal database with password 'test' generated.");
                    }
                }
                catch(IOException ex) {
                    msgError(ex, "Failed to generate new file.", false);
                }
            }
        }
    }

    /**
     * MenuItem About: show Copyright information.
     */
    public class AboutListener implements ActionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        protected AboutListener(KeyringEditor editor) {
            this.editor = editor;
        }

        /**
         * This method shows a copyright information dialog.
         *
         * @param e the ActionEvent to process
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            String[] dbType = {"TripleDES", "TripleDES", "AES128", "AES256"};
            String fileDetails = "";
            if (model != null) {
                fileDetails += "File: " + pdbFilename;
                fileDetails += "\nFile format version: " + model.pdbVersion;
                fileDetails += "\nAlgorithm: " + dbType[model.crypto.type];
                if (model.pdbVersion == 5)
                    fileDetails += "\nIterations: " + model.getIterations();
                fileDetails += "\nEntries: " + model.getEntriesSize();
                fileDetails += "\n\n";
            }
            JOptionPane.showMessageDialog(editor.frame,
                Gui.FRAMETITLE + " " + KeyringEditor.VERSION +
                "\n\nCopyright 2022 Peter Newman\n" +
                "pnewman.com/keyring\n\n" +
                fileDetails +
                "KeyringEditor v" + KeyringEditor.VERSION + " is based on:\n" +
                "KeyringEditor v1.1\n" +
                "Copyright 2005 Markus Griessnig\n" +
                "Vienna University of Technology\n" +
                "Institute of Computer Technology\n\n" +
                "KeyringEditor v1.1 is based on:\n" +
                "Java Keyring v0.6\n" +
                "Copyright 2004 Frank Taylor <keyring@lieder.me.uk>\n\n" +
                "This program is free software; you can redistribute it and/or\n" +
                "modify it under the terms of the GNU General Public License as\n" +
                "published by the Free Software Foundation; either version 3 of\n" +
                "the License, or (at your option) any later version.\n\n" +
                "This program is distributed in the hope that it will be useful,\n" +
                "but WITHOUT ANY WARRANTY; without even the implied warranty of\n" +
                "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n" +
                "See the GNU General Public License for more details.\n",
                "About",
                JOptionPane.INFORMATION_MESSAGE);
        }
    }

    /**
     * MenuItem Quickstart: show quickstart guide.
     */
    public class QuickstartListener implements ActionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        protected QuickstartListener(KeyringEditor editor) {
            this.editor = editor;
        }

        /**
         * This method shows a quickstart guide.
         *
         * @param e the ActionEvent to process
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            if (editor.saveEntry.isEnabled()) {
                msgInformation("Please save the entry first.");
            } else {
                clearEntry();
                quickstartDisplay = true;
                displayQuickstartFile();
                saveEntry.setEnabled(false);
                saveEntry.setBackground(null);
                delEntry.setEnabled(false);
            }
        }
    }
    
    // ----------------------------------------------------------------
    // listener buttons -----------------------------------------------
    // ----------------------------------------------------------------

    // new
    /**
     * Button New: show edit dialog and save new entry to database.
     */
    public class newEntryListener implements ActionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        protected newEntryListener(KeyringEditor editor) {
            this.editor = editor;
        }

        /**
         * This method opens the new entry dialog and and saves the new entry to database.
         *
         * @param e the ActionEvent to process
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            // show edit dialog
            EditDialog editDlg = new EditDialog(editor.frame, editor.model.getCategories());
            editDlg.pack();
            editDlg.setVisible(true);

            Object[] buffer = editDlg.getNewEntry();
            byte[] record = null;
            byte[] ciphertext;
            int recordLength = 0;
            int newEntryId = editor.model.getEntriesSize() + 1;
            int ivLen = 8; // for TripleDES
            int id; // unique id

            try {
                // new entry
                if(buffer[0] != null) {
                    // pack fields to be encrypted into required record format and set IV length
                    switch(model.pdbVersion) {
                        case 4:
                            record = Model.toRecordFormat4( // account + password + notes
                                (String)buffer[2] + "\0" +
                                (String)buffer[3] + "\0" +
                                (String)buffer[4] + "\0");
                            break;
                        case 5:
                            record = Model.toRecordFormat5(
                                (String)buffer[2],
                                (String)buffer[3],
                                (String)buffer[4]);

                            if(model.crypto.type != 1) { // TripleDES
                                ivLen = 16; // AES128, AES256
                            }

                            break;
                    }

                    // encrypt record
                    ciphertext = editor.model.crypto.encrypt(record);

                    byte[] encodedTitle = Model.stringToByteArray((String)buffer[1]);
                    int len = encodedTitle.length + ciphertext.length - 16;
                    switch(model.pdbVersion) {
                        case 4: recordLength = len + 1;	break;
                        case 5:	recordLength = len + 4 + (len % 2) + ivLen;	break;
                    }
                    
                    int maxTextLen = Model.MAX_ENTRY_SIZE - 1;
                    if (recordLength > maxTextLen) {
                        msgAlert("Text exceeds maximum length for an entry of " + maxTextLen + " bytes.");
                        return;
                    }
                    
                    // get new unique id
                    id = editor.model.getNewUniqueId();

                    // new entry object
                    Entry myEntry = new Entry(
                        newEntryId,
                        (String)buffer[1],
                        (Integer)buffer[0],
                        Model.sliceBytes(ciphertext, 16, ciphertext.length - 16),
                            editor.model.crypto,
                            (Integer)buffer[0] | 0x40,
                            id,
                            recordLength,
                            Model.sliceBytes(ciphertext, 0, ivLen)); // Keyring database format 4: IV ignored

                    // register new entry to vector entries
                    editor.model.addEntry((Object)myEntry);

                    // update tree view
                    editor.dynTree.populate();

                    // save database
                    editor.model.saveData(pdbFilename);

                    // mg
                    // show new entry
                    editor.dynTree.show((Object)myEntry);
                }
            }
            catch(Exception ex) {
                msgError(ex, "newEntryListener", true);
            }
        }
    }

    // save
    /**
     * Button Save: save changed entry to database.
     */
    public class saveEntryListener implements ActionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        public saveEntryListener(KeyringEditor editor) {
            this.editor = editor;
        }

        /**
         * This method saves a changed entry to database.
         *
         * @param e the ActionEvent to process
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            saveEntry(editor);
        }
    }

    private void saveEntry(KeyringEditor editor) {
        DefaultMutableTreeNode node = editor.dynTree.getLastNode();
        byte[] record = null;
        byte[] ciphertext;
        int recordLength = 0;
        int ivLen = 8; // for TripleDES

        try {
            // last selected tree node
            if(node != null) {
                // get updated entry
                Entry myEntry = editor.dynTree.getEntry(node);

                char[] temp = editor.currentPassword.getPassword();

                // pack fields to be encrypted into required record format and set IV length
                switch(model.pdbVersion) {
                    case 4:
                        record = Model.toRecordFormat4(
                            editor.currentAccountName.getText() + "\0" + // account + password + notes
                            (new String(temp)) + "\0" +
                            editor.currentNotes.getText() + "\0");
                        break;
                    case 5:
                        record = Model.toRecordFormat5(
                            editor.currentAccountName.getText(),
                            (new String(temp)),
                            editor.currentNotes.getText());

                        if(model.crypto.type != 1) { // TripleDES
                            ivLen = 16; // AES128, AES256
                        }

                        break;
                }
                // encrypt record
                ciphertext = editor.model.crypto.encrypt(record);

                byte[] encodedTitle = Model.stringToByteArray(editor.currentTitle.getText());
                int len = encodedTitle.length + ciphertext.length - 16;
                switch(model.pdbVersion) {
                    case 4: recordLength = len + 1; break;
                    case 5: recordLength = len + 4 + (len % 2) + ivLen; break;
                }

                int maxTextLen = Model.MAX_ENTRY_SIZE - 1;
                if (recordLength > maxTextLen) {
                    msgAlert("Text exceeds maximum length for an entry of " + maxTextLen + " bytes.");
                    return;
                }

                // save changes
                myEntry.title = editor.currentTitle.getText();
                myEntry.encrypted = Model.sliceBytes(ciphertext, 16, ciphertext.length - 16);
                myEntry.category = editor.currentCategory.getSelectedIndex();
                myEntry.attribute = editor.currentCategory.getSelectedIndex() | 0x40; // record ok
                myEntry.recordLength = recordLength;
                myEntry.iv = Model.sliceBytes(ciphertext, 0, ivLen);

                // disable save button to prevent unsaved changes query on updating tree
                saveEntry.setEnabled(false);
                saveEntry.setBackground(null);
                
                // update tree view
                editor.dynTree.populate();

                // save database
                editor.model.saveData(pdbFilename);

                // redisplay modified entry
                editor.dynTree.show(myEntry);

                msgInformation("Entries saved to: " + editor.pdbFilename);
            }
        }
        catch(Exception ex) {
            msgError(ex, "saveEntryListener", true);
        }
    }
    
    // del
    /**
     * Button Delete: delete current entry.
     */
    public class delEntryListener implements ActionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        public delEntryListener(KeyringEditor editor) {
            this.editor = editor;
        }

        /**
         * This method removes a entry from database.
         *
         * @param e the ActionEvent to process
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            DefaultMutableTreeNode node = editor.dynTree.getLastNode();
            Entry myEntry = editor.dynTree.getEntry(node);

            if(node != null) {
                int n = msgConfirm("Are you sure you want to delete entry '" +
                        myEntry.getTitle() + "'?");
                if(n != JOptionPane.YES_OPTION) {
                    return;
                }

                // delete entry
                editor.model.removeEntry((Object)myEntry);

                // update tree view
                editor.dynTree.populate();

                // save changes
                try {
                    editor.model.saveData(pdbFilename);

                    msgInformation("Entry '" + myEntry.getTitle() +
                        "' deleted.\nDatabase file '" + editor.pdbFilename + "' saved.");
                }
                catch(Exception ex) {
                    msgError(ex, "delEntryListener", false);
                }
            }
        }
    }

    // ----------------------------------------------------------------
    // listener -------------------------------------------------------
    // ----------------------------------------------------------------

    // textfields
    /**
     * DocumentListener: check if entry is updated and set button "Save" according.
     */
    public class documentListener implements DocumentListener {
        protected KeyringEditor editor;

        protected documentListener(KeyringEditor editor) {
            this.editor = editor;
        }

        @Override
        public void insertUpdate(DocumentEvent e) {
            updateLog(e, "insert");
        }

        @Override
        public void removeUpdate(DocumentEvent e) {
            updateLog(e, "remove");
        }

        @Override
        public void changedUpdate(DocumentEvent e) {
            updateLog(e, "changed");
        }

        public void updateLog(DocumentEvent e, String action) {
            if(editor.textFieldChanged == false && editor.locked == false
                        && quickstartDisplay == false) {
                editor.textFieldChanged = true;

                editor.saveEntry.setEnabled(true);
                editor.saveEntry.setBackground(Color.YELLOW);
            }
        }
    }

    // tree
    /**
     * TreeSelectionListener: show selected entry.
     */
    public class treeSelectionListener implements TreeSelectionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        protected treeSelectionListener(KeyringEditor editor) {
            this.editor = editor;
        }

        // disable valueChanged() while restoring previous node to save changes
        boolean treeSelectionListenerEnabled = true;
        
        /**
         * This method shows the selected entry.
         *
         * @param e the ActionEvent to process
         */
        @Override
        public void valueChanged(TreeSelectionEvent e) {
            if (!treeSelectionListenerEnabled)
                return;
            if(locked) {
                msgInformation("Please unlock application first.");
                return;
            }
            if (editor.saveEntry.isEnabled()) {
                int n = msgConfirm("There are unsaved changes.\n" +
                    "Would you like to save them?");
                if(n == JOptionPane.YES_OPTION) {
                    // restore previous node, save changes then switch to new entry
                    Entry newEntry = null;
                    DefaultMutableTreeNode node = editor.dynTree.getLastNode();
                    if(node != null && !node.isRoot())
                        newEntry = editor.dynTree.getEntry(node);

                    TreePath oldTreePath = e.getOldLeadSelectionPath();
                    treeSelectionListenerEnabled = false;
                    try {
                        // restore previous node
                        dynTree.getTree().setSelectionPath(oldTreePath);
                    } finally {
                        treeSelectionListenerEnabled = true;
                    }
                    saveEntry(editor);
                    if(newEntry != null) {
                        // restore path to new entry
                        editor.dynTree.show(newEntry);
                    }
                }
            }
            editor.showEntry();
        }
    }

    /**
     * CategorySelectionListener: filter tree view according to selected category.
     */
    public class CategorySelectionListener implements ActionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        public CategorySelectionListener(KeyringEditor editor) {
            this.editor = editor;
        }

        /**
         * This method filters the tree view according to selected category.
         *
         * @param e the ActionEvent to process
         */
        @Override
        public void actionPerformed(ActionEvent e) {

            editor.dynTree.setCategoryFilter(editor.categoryList.getSelectedIndex());

            editor.showEntry();
        }
    }

    /**
     * currentCategorySelectionListener: check if entry category is changed an set button "Save" according.
     */
    public class currentCategorySelectionListener implements ActionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        public currentCategorySelectionListener(KeyringEditor editor) {
            this.editor = editor;
        }

        /**
         * This method sets button "Save" according to state of variable locked and textFieldChanged.
         *
         * @param e the ActionEvent to process
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            //System.out.println(editor.textFieldChanged);

            if(editor.textFieldChanged == false && editor.locked == false
                    && quickstartDisplay == false) {
                editor.textFieldChanged = true;

                editor.saveEntry.setEnabled(true);
                editor.saveEntry.setBackground(Color.YELLOW);
            }
        }
    }

    /**
     * PasswordShowListener: show entry password in clear text according to check box "Hide passwords?"
     */
    public class PasswordShowListener implements ActionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        protected PasswordShowListener(KeyringEditor editor) {
            this.editor = editor;
        }

        /**
         * This method shows password in clear text according to variable showPassword.
         *
         * @param e the ActionEvent to process
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            if(showPassword == false) {
                showPassword = true;
                editor.currentPassword.setEchoChar('\0'); // show password in plaintext
            }
            else {
                showPassword = false;
                editor.currentPassword.setEchoChar('*');
            }
        }
    }

    /**
     * Button Lock / Unlock: set buttons according to variable locked.
     */
    public class PasswordLockListener implements ActionListener {
        protected KeyringEditor editor;

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        protected PasswordLockListener(KeyringEditor editor) {
            this.editor = editor;
        }

        /**
         * This method sets the button "Lock / Unlock" according to variable locked.
         *
         * @param e the ActionEvent to process
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            if(editor.locked == false) {
                editor.setBtnLock(true, true);
                editor.enableButtons(false);
                editor.clearEntry();
            }
            else {
                if(editor.checkPassword() == true) {
                    editor.setBtnLock(false, true);
                    editor.enableButtons(true);
                    editor.showEntry();
                }
            }
        }
    }
    
    // URL pattern matching from Jeff Atwood
    // https://blog.codinghorror.com/the-problem-with-urls/
    private static final String URL_PROTOCOL = "(https?://|ftp://|www\\.)";
    private static final String URL_FILE = "file:///";
    private static final String URL_DOMAIN = "[a-zA-Z0-9]+([-.][a-zA-Z0-9]+)*\\.[a-zA-Z0-9]+([-][a-zA-Z0-9]+)*";
    private static final String URL_TERMINAL_CHARS = "-\\w+&@#%=~()|";
    private static final String URL_NON_TERMINAL_CHARS = "?!:,.;/";
    private static final String URL_CHARS = URL_TERMINAL_CHARS + URL_NON_TERMINAL_CHARS;
    private static final Pattern URL_PATTERN = Pattern.compile("\\(?((" + URL_PROTOCOL + URL_DOMAIN + ")|("
             + URL_FILE + "))([" + URL_CHARS + "]*[" + URL_TERMINAL_CHARS + "])?", Pattern.CASE_INSENSITIVE); 
    
    private void checkForURL() {
        String cardText = currentNotes.getText();
        int len = cardText.length();
        if (len < 4) return;
        int charIndex = currentNotes.getCaretPosition();
        
        // isolate string bounded by space chars
        int endIndex = charIndex;
        if (charIndex == len) --charIndex;
        while (charIndex > 0 && Character.isWhitespace(cardText.charAt(charIndex))) --charIndex;
        while (charIndex >= 0 && !Character.isWhitespace(cardText.charAt(charIndex))) --charIndex;
        ++charIndex;
        while (endIndex < len && !Character.isWhitespace(cardText.charAt(endIndex))) ++endIndex;
        
        if (endIndex - charIndex < 3) return;
        String extract = cardText.substring(charIndex, endIndex);
        
        // match string using a regular expression for valid urls
        final Matcher urlMatcher = URL_PATTERN.matcher(extract);
        if (urlMatcher.find()) {
            String urlStr = urlMatcher.group();
            endIndex = charIndex + urlMatcher.end();
            charIndex += urlMatcher.start();
            if (urlStr.startsWith("(")) {
                // parenthesis is permitted in a url, do the best you can
                ++charIndex;
                if (urlStr.endsWith(")")) --endIndex;
            }
            String finalUrl = cardText.substring(charIndex, endIndex);
            
            // highlight selection
            currentNotes.setCaretPosition(endIndex);
            currentNotes.moveCaretPosition(charIndex);
            currentNotes.getCaret().setSelectionVisible(true);
        
            // Check support for opening browser
            if (!Desktop.isDesktopSupported())
                return;
            Desktop desktop = Desktop.getDesktop();
            if (!desktop.isSupported(Desktop.Action.BROWSE))
                return;
        
            // launch browser
            URI uri = null;
            try {
                uri = new URI(finalUrl);
                desktop.browse(uri);
            } catch (IOException | URISyntaxException ex) {
                msgError(ex, "Launch browser url: \"" + uri + "\"", false);
            }
        }
    }
    
}
