/*
KeyringEditor

Copyright 2004 Markus Griessnig
Vienna University of Technology
Institute of Computer Technology

KeyringEditor is based on:
Java Keyring v0.6
Copyright 2004 Frank Taylor <keyring@lieder.me.uk>

These programs are distributed in the hope that they will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.
 */

// Gui.java

// 11.11.2004

// 15.11.2004: newEntry, saveEntry, delEntry
// 17.11.2004: new parameter filename (keyring database); edit notes, edit categories
// 18.11.2004: changed currentCategory to ComboBox; changed Layout; JPasswordField
// 22.11.2004: added DynamicTree; added DocumentListener
// 23.11.2004: currentCategorySelectionListener; PasswordTimeoutWorker
// 24.11.2004: CategoriesDialog;
// 25.11.2004: split class into Gui.java and Editor.java
// 05.12.2004: MenuItem Tools - Convert database added
// 13.05.2018: Changed password lock timer default to 10 minutes
// 13.05.2018: Password lick timeout clears screen and sets gui buttons
// 26.06.2021: Add text wrap to notes panel

package com.pnewman.apps.keyring;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.KeyEvent;
import java.util.Date;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;

/**
 * This class setup the gui and the password timeout functions.
 */
public class Gui {
    // ----------------------------------------------------------------
    // variables
    // ----------------------------------------------------------------
    protected static final String VERSION = "1.2";
    protected static final String FRAMETITLE = "KeyringEditor";
    /**
     * Time in milliseconds till password timeout.
     */
    protected static long PASSWORD_TIMEOUT = 1200000; // default: 20 minutes = 20 * 60 s * 1000 ms

    // MenuBar
    protected JMenuBar menuBar;
    // File
    protected JMenuItem openMenuItem;
    protected JMenuItem closeMenuItem;
    protected JMenuItem quitMenuItem;
    // Tools
    protected JMenuItem categoriesMenuItem;
    protected JMenuItem csvMenuItem;
    protected JMenuItem convertMenuItem;
    protected JMenuItem newMenuItem;

    // Help
    protected JMenuItem aboutMenuItem;
    protected JMenuItem quickstartMenuItem;

    // entryListPane
    protected JComboBox categoryList;
    protected DynamicTree dynTree;

    // entryPane
    protected JComboBox currentCategory;
    protected JTextField currentTitle;
    protected JTextField currentAccountName;
    protected JPasswordField currentPassword;
    protected JTextField currentDate;
    protected JTextArea currentNotes;
    protected JButton saveEntry;

    // buttonPane
    protected JButton newEntry;
    protected JButton delEntry;
    protected JButton btnLock;
    protected JCheckBox currentPasswordShow;
    
    // Password Timeout -----------------------------------------------
    /**
     * This class handles the password timeout.
     * It is started by class KeyringEditor as a thread.
     */
    protected class PasswordTimeoutWorker implements Runnable {
        /**
         * Reference to class Editor
         */
        private final KeyringEditor editor;

        /**
         * Time when password timeout starts plus user defined password timeout
         */
        private Date endDate = null;

        private final Object sleepGate = new Object();

        /**
         * Default constructor.
         *
         * @param editor Reference to class Editor
         */
        public PasswordTimeoutWorker(KeyringEditor editor) {
            this.editor = editor;
        }

        /**
         * Sets the variable endDate to new timeout.
         */
        public synchronized void restartTimeout() {
            endDate = new Date((new Date()).getTime() + PASSWORD_TIMEOUT);
//            System.out.println("timeout reset = " + PASSWORD_TIMEOUT);

            synchronized (sleepGate) {
                sleepGate.notifyAll();
            }
        }

        /**
         * Sets the variable endDate to null and forces a timeout.
         */
        public synchronized void setTimeout() {
            endDate = null;
        }

        /**
         * Returns variable endDate.
         * @return 
         */
        public synchronized Date getEndDate() {
            return endDate;
        }

        /**
         * Timeout thread.
         */
        @Override
        @SuppressWarnings("empty-statement")
        public void run() {
            while(true) {
                if(endDate != null) {
                    Date now = new Date();

                    long difference = endDate.getTime() - now.getTime();

                    // timed out
                    if(difference <= 0) {
                        editor.lockAndClearScreen();
                        endDate = null;
                    }

                    try {
                        Thread.sleep(1000); // sleep 1 s
                    }
                    catch (InterruptedException ignore) {};
                }
                else { // timed out
                        try {
                            synchronized (sleepGate) {
                                sleepGate.wait();
                            }
                        }
                        catch (InterruptedException e) {
                        }
                }
            }
        }
    }

    // ----------------------------------------------------------------
    // protected ------------------------------------------------------
    // ----------------------------------------------------------------

    // setMenuBar -----------------------------------------------------
    /**
     * Setup menubar.
     * @return 
     */
    protected JMenuBar setMenuBar() {
    // Function: creates menu bar
    // Parameters: -
    // Returns: JMenubar
        menuBar = new JMenuBar();
        //frame.setJMenuBar(menuBar);

        JMenu fileMenu = new JMenu("File");
        fileMenu.setMnemonic(KeyEvent.VK_F);
        menuBar.add(fileMenu);

        openMenuItem = new JMenuItem("Open", KeyEvent.VK_O);
        fileMenu.add(openMenuItem);

        closeMenuItem = new JMenuItem("Close", KeyEvent.VK_C);
        fileMenu.add(closeMenuItem);
        closeMenuItem.setEnabled(false);

        fileMenu.addSeparator();

        quitMenuItem = new JMenuItem("Quit", KeyEvent.VK_Q);
        fileMenu.add(quitMenuItem);

        JMenu catMenu = new JMenu("Tools");
        catMenu.setMnemonic(KeyEvent.VK_T);
        menuBar.add(catMenu);

        categoriesMenuItem = new JMenuItem("Edit categories", KeyEvent.VK_E);
        catMenu.add(categoriesMenuItem);

        catMenu.addSeparator();

        csvMenuItem = new JMenuItem("Save to CSV file", KeyEvent.VK_S);
        catMenu.add(csvMenuItem);

        convertMenuItem = new JMenuItem("Convert database", KeyEvent.VK_C);
        catMenu.add(convertMenuItem);

        newMenuItem = new JMenuItem("New database file", KeyEvent.VK_N);
        catMenu.add(newMenuItem);

        JMenu helpMenu = new JMenu("Help");
        helpMenu.setMnemonic(KeyEvent.VK_H);
        menuBar.add(helpMenu);

        aboutMenuItem = new JMenuItem("About", KeyEvent.VK_A);
        helpMenu.add(aboutMenuItem);

        quickstartMenuItem = new JMenuItem("Quickstart", KeyEvent.VK_Q);
        helpMenu.add(quickstartMenuItem);

        return menuBar;
    }

    // setLayout ------------------------------------------------------
    /**
     * Setup gui layout.
     * @param editor
     * @return 
     */
    protected JSplitPane setLayout(KeyringEditor editor) {
    // Function: creates gui elements (GridBagLayout)
    // Parameters: -
    // Returns: JSplitPane
        int gridy = 0;

        // Gui Elements
        categoryList = new JComboBox();
        dynTree = new DynamicTree(editor);

        currentCategory = new JComboBox();
        currentTitle = new JTextField();
        currentAccountName = new JTextField();
        currentPassword = new JPasswordField();
        currentDate = new JTextField();
        currentDate.setEnabled(false); // date is not editable
        currentNotes = new JTextArea();
        currentNotes.setLineWrap(true);
        currentNotes.setWrapStyleWord(true);
/*
        Font font = currentNotes.getFont();
        if (font.getSize() == 12) {
            // slightly larger font for the notes panel on a Windows system
            currentNotes.setFont(new Font(font.getName(), font.getStyle(), 13));
        }
*/
        currentPassword.setEchoChar('\0');		// default show passwd
        currentPasswordShow = new JCheckBox("Hide Passwords?", false);
        btnLock = new JButton("Lock");
        newEntry = new JButton("New Entry");
        saveEntry = new JButton("Save");
        delEntry = new JButton("Delete");

        // Panes
        JPanel entryListPane = new JPanel();
        entryListPane.setBorder(BorderFactory.createLineBorder(Color.black));

        JPanel entryPane = new JPanel();
        entryPane.setBorder(BorderFactory.createLineBorder(Color.black));

        JPanel buttonPane = new JPanel();
        buttonPane.setBorder(BorderFactory.createLineBorder(Color.black));

        // Entry-List-Pane --------------------------------------------
        GridBagLayout gridbag1 = new GridBagLayout();
        GridBagConstraints c = new GridBagConstraints();
        entryListPane.setLayout(gridbag1);

        c.ipadx = 2;
        c.ipady = 2;
        c.insets = new Insets(2, 2, 2, 2);
        c.anchor = GridBagConstraints.NORTHWEST;
        c.gridx = 0;
        c.gridy = 0;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.weightx = 0;
        c.weighty = 0;

        gridbag1.setConstraints(categoryList, c);
        entryListPane.add(categoryList);

        c.gridy = 1;
        c.fill = GridBagConstraints.BOTH;
        c.weightx = 1.0;
        c.weighty = 1.0;

        gridbag1.setConstraints(dynTree, c);
        entryListPane.add(dynTree);

        // Entry-Pane -------------------------------------------------
        GridBagLayout gridbag2 = new GridBagLayout();
        entryPane.setLayout(gridbag2);

        c.ipadx = 2;
        c.ipady = 2;
        c.insets = new Insets(2, 2, 2, 2);
        c.anchor = GridBagConstraints.NORTHWEST;
        c.gridwidth = 1;
        c.gridheight = 1;
        c.weightx = 0.0;
        c.weighty = 0.0;

        // category
        JLabel categoryLabel = new JLabel("Category: ");
        c.gridx = 0;
        c.gridy = gridy;
        c.gridwidth = 1;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.EAST;
        c.weightx = 0.0;
        gridbag2.setConstraints(categoryLabel, c);
        entryPane.add(categoryLabel);

        c.gridx = 1;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor = GridBagConstraints.NORTHWEST;
        c.weightx = 1.0;
        gridbag2.setConstraints(currentCategory, c);
        entryPane.add(currentCategory);

        // title
        JLabel titleLabel = new JLabel("Title: ");
        c.gridx = 0;
        c.gridy = ++gridy;
        c.gridwidth = 1;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.EAST;
        c.weightx = 0.0;
        gridbag2.setConstraints(titleLabel, c);
        entryPane.add(titleLabel);

        c.gridx = 1;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor = GridBagConstraints.NORTHWEST;
        c.weightx = 1.0;
        gridbag2.setConstraints(currentTitle, c);
        entryPane.add(currentTitle);

        // account
        JLabel accountNameLabel = new JLabel("Account: ");
        c.gridx = 0;
        c.gridy = ++gridy;
        c.gridwidth = 1;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.EAST;
        c.weightx = 0.0;
        gridbag2.setConstraints(accountNameLabel, c);
        entryPane.add(accountNameLabel);

        c.gridx = 1;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor = GridBagConstraints.NORTHWEST;
        c.weightx = 1.0;
        gridbag2.setConstraints(currentAccountName, c);
        entryPane.add(currentAccountName);

        // password
        JLabel passwordLabel = new JLabel("    Password: ");
        c.gridx = 0;
        c.gridy = ++gridy;
        c.gridwidth = 1;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.EAST;
        c.weightx = 0.0;
        gridbag2.setConstraints(passwordLabel, c);
        entryPane.add(passwordLabel);

        c.gridx = 1;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor = GridBagConstraints.NORTHWEST;
        c.weightx = 1.0;
        gridbag2.setConstraints(currentPassword, c);
        entryPane.add(currentPassword);

        // date
        JLabel dateLabel = new JLabel("Date: ");
        c.gridx = 0;
        c.gridy = ++gridy;
        c.gridwidth = 1;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.EAST;
        c.weightx = 0.0;
        gridbag2.setConstraints(dateLabel, c);
        entryPane.add(dateLabel);

        c.gridx = 1;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor = GridBagConstraints.NORTHWEST;
        c.weightx = 1.0;
        gridbag2.setConstraints(currentDate, c);
        entryPane.add(currentDate);

        // notes
        c.gridx = 0;
        c.gridy = ++gridy;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.fill = GridBagConstraints.BOTH;
        c.anchor = GridBagConstraints.NORTHWEST;
        c.weightx = 1.0;
        c.weighty = 1.0;

        JScrollPane currentNotesScroll = new JScrollPane(currentNotes);
        gridbag2.setConstraints(currentNotesScroll, c);
        entryPane.add(currentNotesScroll);

        // save
        c.gridx = 0;
        c.gridy = ++gridy;
        c.gridwidth = 1;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor = GridBagConstraints.NORTHWEST;
        c.weightx = 0.0;
        c.weighty = 0.0;
        gridbag2.setConstraints(saveEntry, c);
        entryPane.add(saveEntry);

        // buttonPane
        //FlowLayout flow = new FlowLayout(FlowLayout.LEADING);
        //BoxLayout flow = new BoxLayout(BoxLayout.X_AXIS);
        //buttonPane.setLayout(flow);
        buttonPane.setLayout(new BoxLayout(buttonPane, BoxLayout.X_AXIS));

        buttonPane.add(newEntry);
        buttonPane.add(Box.createRigidArea(new Dimension(5,0)));
        buttonPane.add(delEntry);
        buttonPane.add(Box.createRigidArea(new Dimension(5,0)));
        buttonPane.add(btnLock);
        buttonPane.add(Box.createRigidArea(new Dimension(5,0)));
        buttonPane.add(currentPasswordShow);

        // JSplitPane -------------------------------------------------
        JSplitPane top = new JSplitPane(JSplitPane.VERTICAL_SPLIT, entryPane, buttonPane);
        top.setResizeWeight(1);

        JSplitPane contentPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, entryListPane, top);
        contentPane.setResizeWeight(0.33);

        entryListPane.setMinimumSize(new Dimension(200, 440));
        entryPane.setMinimumSize(new Dimension(300, 400));
        buttonPane.setMinimumSize(new Dimension(300, 40));

//        entryListPane.setPreferredSize(new Dimension(150, 340));
//        entryPane.setPreferredSize(new Dimension(450, 300));
//        buttonPane.setPreferredSize(new Dimension(450, 40));

        entryListPane.setPreferredSize(new Dimension(200, 440));
        entryPane.setPreferredSize(new Dimension(500, 400));
        buttonPane.setPreferredSize(new Dimension(500, 40));

        //frame.setContentPane(contentPane);
        return contentPane;
    }
}
