/*
KeyringEditor

Copyright 2004 Markus Griessnig
Vienna University of Technology
Institute of Computer Technology

KeyringEditor is based on:
Java Keyring v0.6
Copyright 2004 Frank Taylor <keyring@lieder.me.uk>

These programs are distributed in the hope that they will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.
 */

// Entry.java

// 29.10.2004

// 02.11.2004: add variables attribute, uniqueId & recordLength
// 06.11.2004: removed entryId from crypto.decrypt(); added getDate()
// 17.11.2004: added getAll()
// 23.11.2004: toString changed
// 24.11.2004: added setTitleSeparator()
// 01.12.2004: added iv (Keyring database format 5)
// 07.12.2004: using lastIndexOf for toString
// 23.09.2005: added getUniqueId()

package com.pnewman.apps.keyring;

// Keyring fields: title, category, account, password, notes, date

/**
 * This class is used to save and manipulate entries.
 */
public class Entry implements Comparable<Entry> {
    // ----------------------------------------------------------------
    // variables
    // ----------------------------------------------------------------

    /**
     * Separates levels in an entry title for the tree view
     */
    private static char SEPARATOR = '/'; // default

    /**
     * Entry Id
     */
    protected int entryId;

    /**
     * Entry Title
     */
    protected String title;

    /**
     * Index of category-name
     */
    protected int category;

    /**
     * Encrypted data: account, password, notes, date
     */
    protected byte[] encrypted;

    /**
     * IV-Vector of encrypted data
     */
    protected byte[] iv;

    /**
     * Contains Index of category-name and information about if record is hidden
     */
    protected int attribute;

    /**
     * Used by keyring database
     */
    protected int uniqueId;

    /**
     * Length of entry record in keyring database
     */
    protected int recordLength;

    /**
     * Reference to class Crypto
     */
    private final Crypto crypto; // reference

    // ----------------------------------------------------------------
    // constructor
    // ----------------------------------------------------------------
    /**
     * Default constructor.
     * @param entryId
     * @param title
     * @param category
     * @param encrypted
     * @param crypto
     * @param attribute
     * @param uniqueId
     * @param recordLength
     * @param iv
     */
    public Entry(int entryId, String title, int category, byte[] encrypted, Crypto crypto,
        int attribute, int uniqueId, int recordLength, byte[] iv) {
        this.entryId = entryId;
        this.title = title;
        this.category = category;
        this.encrypted = encrypted;
        this.crypto = crypto;
        this.attribute = attribute;
        this.uniqueId = uniqueId;
        this.recordLength = recordLength;
        this.iv = iv; // null if Keyring database format 4
    }

    // ----------------------------------------------------------------
    // public ---------------------------------------------------------
    // ----------------------------------------------------------------

    /**
     * This method returns the unique id.
     *
     * @return Unique id
     */
    public int getUniqueId() {
        return uniqueId;
    }

    /**
     * This method sets the variable SEPARATOR.
     *
     * @param sep Entry title separator (Default: '/')
     */
    public void setTitleSeparator(char sep) {
        SEPARATOR = sep;
    }

    // uncrypted fields

    /**
     * This method returns the entry id.
     *
     * @return Entry id
     */
    public int getEntryId() {
        return entryId;
    }

    /**
     * This method returns the entry title.
     *
     * @return Entry title
     */
    public String getTitle() {
        return title;
    }

    /**
     * This method returns the category-name index.
     *
     * @return Category-name index
     */
    public int getCategory() {
        return category;
    }

    // crypted fields

    /**
     * This method returns the decrypted account name.
     *
     * @return Account name
     */
    public String getAccount() {
        String temp = (String)crypto.decrypt(encrypted, "account", iv);

        if(temp == null)
            return "";
        else
            return temp;
    }

    /**
     * This method returns the decrypted password.
     *
     * @return Password
     */
    public String getPassword() {
        String temp = (String)crypto.decrypt(encrypted, "password", iv);

        if(temp == null)
            return "";
        else
            return temp;
    }

    /**
     * This method returns the decrypted notes.
     *
     * @return Notes
     */
    public String getNotes() {
        String temp = (String)crypto.decrypt(encrypted, "notes", iv);

        if(temp == null)
            return "";
        else
            return temp;
    }

    /**
     * This method returns the decrypted last modified date.
     *
     * @return Date in format dd.mm.yyy
     */
    public String getDate() {
        byte[] buffer = (byte[])crypto.decrypt(encrypted, "datetype", iv);
        int d, m, y;
        //Model.printHexByteArray("getDate", buffer);

        try {
            y = ((buffer[0] & 0xFE) >> 1) + 1904;
            m = ((buffer[0] & 0x01) << 3) + ((buffer[1] & 0xE0) >> 5);
            d = (buffer[1] & 0x1F);
        }
        catch(Exception e) {
            return "11.11.2004"; // mg: because of wrong AES encryption in keyring pre-release
        }

        return d + "." + m + "." + y;
    }

    // for testing purpose
    public byte[] getAll() {
        byte[] buffer = (byte [])crypto.decrypt(encrypted, "", iv);

        return buffer;
    }

    /**
     * This method returns a string with all entry information.
     *
     * @return Entry information
     */
    public String getInfo() {
        return "EntryId: " + entryId + " = " + title + " + " + getAccount() + " + " + getPassword() +
                " (" + recordLength + ", " + getDate() + ", " + getCategory() + ")";
    }

    // toString used by DefaultTreeModel in DynamicTree.java
    /**
     * This method is used by DefaultTreeModel in DynamicTree.java to display the entry title.
     * It uses variable SEPARATOR to get the last level of the title.
     * Example: Labor1/PC1 => PC1
     * @return 
     */
    @Override
    public String toString() {
        int i = title.lastIndexOf(SEPARATOR);

        if(i != -1) {
            return title.substring(i + 1, title.length());
        }
        else {
            return title;
        }
    }

    // comparable for sorting entries (title)
    @Override
    public int compareTo(Entry e) {
        return (title.toLowerCase()).compareTo((e.getTitle()).toLowerCase());
    }
}
