/*
KeyringEditor

Copyright 2004 Markus Griessnig
Vienna University of Technology
Institute of Computer Technology

KeyringEditor is based on:
Java Keyring v0.6
Copyright 2004 Frank Taylor <keyring@lieder.me.uk>

These programs are distributed in the hope that they will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.
*/

// ConvertDialog.java

// 07.12.2004
// 26.01.2005: check for empty password
// 13.05.2018: added 10,000 option to number of iterations dropdown box
// 04.09.2021: changed default values for cipher and iterations combo boxes

package com.pnewman.apps.keyring;

import java.awt.event.*;
import javax.swing.*;
import java.beans.*; // property change stuff
import java.util.*;
import java.awt.*;
import java.io.*;
import javax.swing.filechooser.FileNameExtensionFilter;

/**
 * This dialog allows the user to convert a database to different keyring database formats.
 */
public class ConvertDialog extends JDialog implements ActionListener, PropertyChangeListener {
    // ----------------------------------------------------------------
    // variables
    // ----------------------------------------------------------------
    /**
     * Reference to the Gui frame
     */
    private final Frame frame;

    private final JComboBox cboCipher;
    private final JComboBox cboIter;
    private JPasswordField password;

    /**
     * Reference to the class Model
     */
    private final Model model;

    /**
     * Database format of the loaded database
     */
    private final int fromFormat;

    private JOptionPane optionPane;

    private final String btnString1 = "OK";
    private final String btnString2 = "Cancel";

    /**
     * True if button "Cancel" pressed, otherwise false
     */
    private boolean cancelled = false;

    // ----------------------------------------------------------------
    // constructor
    // ----------------------------------------------------------------
    /**
     * Default constructor generates Dialog.
     *
     * @param frame Reference to the Gui frame
     * @param model Reference to class Model
     * @param fromFormat Database format of the loaded database
     */
    public ConvertDialog(Frame frame, Model model, int fromFormat) {
        super(frame, "Convert database", true);

        this.frame = frame;
        this.model = model;
        this.fromFormat = fromFormat;

        // Cipher
        ArrayList<String> cipher = new ArrayList<>();
        cipher.add("TripleDES | (insecure) format 4");
        cipher.add("TripleDES | (insecure) format 5");
        cipher.add("AES128 | Database format 5");
        cipher.add("AES256 | Database format 5");

        cboCipher = new JComboBox(cipher.toArray());
        cboCipher.setModel(new DefaultComboBoxModel(cipher.toArray()));
        cboCipher.setSelectedIndex(2);

        // iterations
        ArrayList<Integer> iter = new ArrayList<>();
        iter.add(50);
        iter.add(250);
        iter.add(500);
        iter.add(1000);
        iter.add(10000);
        
        cboIter = new JComboBox(iter.toArray());
        cboIter.setModel(new DefaultComboBoxModel(iter.toArray()));
        cboIter.setSelectedIndex(4);

        password = new JPasswordField(20);

        // labels
        String msgString1 = "Cipher: ";
        String msgString2 = "Iterations: ";
        String msgString3 = "New password: ";

        Object[] array = {msgString1, cboCipher, msgString2, cboIter, msgString3, password};

        Object[] options = {btnString1, btnString2};

        // generate dialog
        optionPane = new JOptionPane(array,
            JOptionPane.PLAIN_MESSAGE,
            JOptionPane.YES_NO_OPTION,
            null,
            options,
            options[0]);

        setContentPane(optionPane);

        setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
        addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent we) {
            /*
             * Instead of directly closing the window,
             * we're going to change the JOptionPane's
             * value property.
             */
             optionPane.setValue(JOptionPane.CLOSED_OPTION);
            }
        });

        addComponentListener(new ComponentAdapter() {
            @Override
            public void componentShown(ComponentEvent ce) {
                    password.requestFocusInWindow();
            }
        });

        //Register an event handler that puts the text into the option pane.
        //allCategories.addActionListener(this);

        //Register an event handler that reacts to option pane state changes.
        optionPane.addPropertyChangeListener(this);
    }

    // ----------------------------------------------------------------
    // public ---------------------------------------------------------
    // ----------------------------------------------------------------

    /**
     * This method returns status of variable canceled.
     *
     * @return variable canceled
     */
    public boolean getCancelled() {
        return cancelled;
    }

    /**
     * This method is empty.
     *
     * @param e the ActionEvent to process
     */
    @Override
    public void actionPerformed(ActionEvent e) {
    }

    /**
     * This method processes the pressed button.
     * If button is OK database is converted to selected keyring database format and
     * encrypted with new password and selected cryptographic function.
     * Otherwise variable canceled is set to true.
     *
     * @param e the PropertyChangeEvent to process
     */
    @Override
    public void propertyChange(PropertyChangeEvent e) {
        String prop = e.getPropertyName();

        if (isVisible()
            && (e.getSource() == optionPane)
            && (JOptionPane.VALUE_PROPERTY.equals(prop) ||
                    JOptionPane.INPUT_VALUE_PROPERTY.equals(prop))) {

            Object value = optionPane.getValue();

            if(value == JOptionPane.UNINITIALIZED_VALUE) {
                //ignore reset
                return;
            }

            //Reset the JOptionPane's value.
            //If you don't do this, then if the user
            //presses the same button next time, no
            //property change event will be fired.
            optionPane.setValue(JOptionPane.UNINITIALIZED_VALUE);

            if(btnString1.equals(value)) {
                // check for empty password
                if(password.getPassword().length == 0) {
                    JOptionPane.showMessageDialog(ConvertDialog.this, "Sorry, password must not be empty!\n",
                        "Error", JOptionPane.ERROR_MESSAGE);
                    cancelled = true;
                    password.requestFocusInWindow();
                    return;
                }

                // ok
                JFileChooser chooser = new JFileChooser();

                chooser.setDialogTitle("Convert Keyring database");
                chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
                chooser.setMultiSelectionEnabled(false);
                FileNameExtensionFilter filter = new FileNameExtensionFilter("Keyring '.pdb' file", "pdb");
                chooser.setFileFilter(filter);

                int returnVal = chooser.showSaveDialog(this.frame);

                if(returnVal == JFileChooser.APPROVE_OPTION) {
                    try {
                        File selectedFile = chooser.getSelectedFile();
                        String toFilename = selectedFile.getCanonicalPath();
                        if(!toFilename.toLowerCase().endsWith(".pdb")) {
                            toFilename += ".pdb";
                        }

                        int toFormat = 0;
                        int type = cboCipher.getSelectedIndex();
                        // 0 = TripleDES Format 4
                        // 1 = TripleDES Format 5
                        // 2 = AES-128 Format 5
                        // 3 = AES-256 Format 5

                        switch(type) {
                                case 0: toFormat = 4; break;
                                case 1:
                                case 2:
                                case 3: toFormat = 5; break;
                        }

                        // convert database
                        model.convertDatabase(fromFormat,
                            toFormat,
                            toFilename,
                            password.getPassword(),
                            type,
                            ((Integer)cboIter.getSelectedItem()));
                    }
                    catch(Exception ex) {
                        // cancel
                        cancelled = true;

                        JOptionPane.showMessageDialog(frame,
                                "Convert database: " + ex.getMessage(),
                                "Error",
                                JOptionPane.ERROR_MESSAGE);

                        ex.printStackTrace(System.err);
                    }
                }
                else {
                    // cancel
                    cancelled = true;
                }
            }
            else {
                // cancel
                cancelled = true;
            }

            clearAndHide();
        }
    }

    // ----------------------------------------------------------------
    // private --------------------------------------------------------
    // ----------------------------------------------------------------

    /**
     * This method hides the dialog.
     */
    private void clearAndHide() {
        password.setText(null);
        setVisible(false);
    }
}
