/*
KeyringEditor

Copyright 2004 Markus Griessnig
Vienna University of Technology
Institute of Computer Technology

KeyringEditor is based on:
Java Keyring v0.6
Copyright 2004 Frank Taylor <keyring@lieder.me.uk>

These programs are distributed in the hope that they will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.
*/

// Editor.java

// 25.11.2004

// 01.12.2004: Keyring database format 5 support added
// 05.12.2004: MenuItem Tools - Convert database added
// 12.01.2004: Model.writeNewDatabase() in main() added
// 24.05.2005: showEntry - check no category

import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.tree.*;
import java.awt.event.*;
import javax.swing.*;
import java.beans.*; // property change stuff
import java.util.*;
import java.awt.*;
import java.io.*;

/**
 * This class handles the gui.
 */
public class Editor extends Gui {
	// ----------------------------------------------------------------
	// variables
	// ----------------------------------------------------------------
	/**
	 * Separates levels in an entry title for the tree view
	 */
	protected char SEPARATOR = '/'; // entry title separator

	/**
	 * Last directory to load from
	 */
	private File previousDirectory = null; // last directory to load from

	/**
	 * Current loaded keyring database
	 */
	private String pdbFilename = "dummy.pdb"; // default database to save to

	private Gui gui;
	private Model model;
	private JFrame frame;
	private PasswordTimeoutWorker timeoutThread;

	// flags
	/**
	 * Category-filter (0 = show all)
	 */
	private int filterCategory = -1;

	/**
	 * Show button "Save" only when entry text changes (true)
	 */
    private boolean textFieldChanged = true; // show button save only when text changes

	/**
	 * Show entry password in clear text (true)
	 */
	private boolean showPassword = false;

	/**
	 * True when application is locked
	 */
    private boolean locked = false;

	// ----------------------------------------------------------------
	// main -----------------------------------------------------------
	// ----------------------------------------------------------------
	/**
	 * main
	 *
	 * @param argv Commandline parameters
	 */
	public static void main(String[] argv) {
		String dbFilename = null;

		Editor myEditor = new Editor();

		myEditor.frame = new JFrame(FRAMETITLE);
		myEditor.frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

		// check command line parameters
		if(argv.length > 1) {
			System.out.println("Usage: java -jar KeyringEditor.jar [keyring-database]");
			return;
		}

		if(argv.length > 0) {
			dbFilename = argv[0];
		}

		// setup gui
		try {
			myEditor.setupGui(dbFilename);
		}
		catch(Exception e) {
			myEditor.msgError(e, "main", true);
		}
	}

	// ----------------------------------------------------------------
	// public ---------------------------------------------------------
	// ----------------------------------------------------------------

	/**
	 * Returns filename of the current loaded keyring database.
	 *
	 * @return Filename
	 */
	public String getFilename() {
		return this.pdbFilename;
	}

	/**
	 * Returns separator for title levels.
	 *
	 * @return Separator
	 */
	public char getSeparator() {
		return this.SEPARATOR;
	}

	/**
	 * Returns reference to class Model.
	 *
	 * @return Reference to class Model
	 */
	public Model getModel() {
		return this.model;
	}

	// ----------------------------------------------------------------
	// private --------------------------------------------------------
	// ----------------------------------------------------------------

	/**
	 * Checks if a file already exists and show warning dialog.
	 *
	 * @param filename Filename
	 *
	 * @return False if file operation should be cancelled
	 */
	private boolean checkFile(String filename) {
		boolean ok = true;

		File f = new File(filename);

		if(f.exists() == true) {
			int n = JOptionPane.showConfirmDialog(
				frame, "File already exists. Continue?",
				"Warning",
				JOptionPane.YES_NO_OPTION);

			if(n == JOptionPane.NO_OPTION) {
				ok = false;
			}
		}

		return ok;
	}

	/**
	 * Shows a error message.
	 *
	 * @param e Exception
	 * @param info User-defined text
	 * @param showStack True if Exception Stack Trace should be displayed
	 */
	private void msgError(Exception e, String info, boolean showStack) {
		JOptionPane.showMessageDialog(frame,
			info + ": " + e.getMessage(),
			"Error",
			JOptionPane.ERROR_MESSAGE);

		if(showStack) {
			e.printStackTrace(System.err);
		}
	}

	/**
	 * Shows a information message.
	 *
	 * @param info User-defined text
	 */
	private void msgInformation(String info) {
		JOptionPane.showMessageDialog(frame,
			info,
			"Information",
			JOptionPane.INFORMATION_MESSAGE);
	}


	// setupGui -------------------------------------------------------
	/**
	 * Loads menubar, adds ActionListeners, starts PasswordTimeout Thread.
	 *
	 * @param dbFilename Keyring database or null
	 */
	private void setupGui(String dbFilename) throws Exception {
	// Function: setup gui
	// Parameters: keyring-database
	// Returns: -

		// MenuBar
		JMenuBar myMenuBar = setMenuBar();
		frame.setJMenuBar(myMenuBar);

		// Layout
		JSplitPane mySplitPane = setLayout(this);
		frame.setContentPane(mySplitPane);

		// MenuBar Listener
		openMenuItem.addActionListener(new OpenListener(this));
		closeMenuItem.addActionListener(new CloseListener(this));
		quitMenuItem.addActionListener(new QuitListener(this));
		categoriesMenuItem.addActionListener(new editCategoriesListener(this));
		csvMenuItem.addActionListener(new csvListener(this));
		aboutMenuItem.addActionListener(new AboutListener(this));
		convertMenuItem.addActionListener(new convertListener(this));
		newMenuItem.addActionListener(new newListener(this));

		// entryPane Listener
		currentCategory.addActionListener(new currentCategorySelectionListener(this));
		currentTitle.getDocument().addDocumentListener(new documentListener(this));
		currentAccountName.getDocument().addDocumentListener(new documentListener(this));
		currentPassword.getDocument().addDocumentListener(new documentListener(this));
		currentNotes.getDocument().addDocumentListener(new documentListener(this));

		// entryListPane Listener
		categoryList.addActionListener(new CategorySelectionListener(this));
		dynTree.getTree().addTreeSelectionListener(new treeSelectionListener(this));

		// buttonPane Listener
		newEntry.addActionListener(new newEntryListener(this));
		saveEntry.addActionListener(new saveEntryListener(this));
		delEntry.addActionListener(new delEntryListener(this));
		btnLock.addActionListener(new PasswordLockListener(this));
		currentPasswordShow.addActionListener(new PasswordShowListener(this));

		// Frame
		frame.pack();
		frame.setVisible(true);

		// Passwort Timeout
		timeoutThread = new PasswordTimeoutWorker(this);
		new Thread(timeoutThread).start();

		// load Database
		loadDatabase(dbFilename);
	}

	// loadDatabase ---------------------------------------------------
	/**
	 * Loads a Keyring database and setup gui (buttons, menubar) properly.
	 *
	 * @param dbFilename Keyring database or null
	 */
	private void loadDatabase(String dbFilename) throws Exception {
	// Function: load data model
	// Parameters: keyring-database
	// Returns: -
		String[] dbType = {"TripleDES", "TripleDES", "AES128", "AES256"};

		if(dbFilename != null) {
			model = new Model();

			try {
				model.loadData(dbFilename);

				pdbFilename = dbFilename;

				setupProperties(model.getElements());
			}
			catch(Exception ex) {
				msgError(ex, "Open keyring database", false);

				try {
					loadDatabase(null);
					return;
				}
				catch(Exception ignore) {};
			}

			// Password dialog
			if(checkPassword() == false) {
				loadDatabase(null);
				return;
			}

			// initialize buttons etc.
			frame.setTitle(FRAMETITLE + ": " + dbFilename + " (" + dbType[model.crypto.type] + " | Database format " + model.pdbVersion + ")");
			//setupCategories(this.model.getCategories()); // Java 1.5
			setupCategories((Vector)this.model.getCategories());
			setMenuBar(false);
			enableButtons(true);
			setBtnLock(false, true);
			dynTree.populate();
		}
		else {
			model = null;

			// initialize buttons etc.
			frame.setTitle(FRAMETITLE);
	     	setupCategories(null);
	     	setMenuBar(true);
	     	enableButtons(false);
			setBtnLock(false, false);
			dynTree.clear();
		}

		//frame.show();
	}

	/**
	 * Set level separator to display correct entry title.
	 *
	 * @param entries Keyring entries
	 */
	private void setupProperties(Enumeration entries) {
		// set properties
		Prop myProp = new Prop(this);
		myProp.setup();

		// set separator in entry objects
		for(Enumeration e = entries; e.hasMoreElements(); ) {
			Entry entry = (Entry)e.nextElement();
			entry.setTitleSeparator(SEPARATOR);
		}
	}

	/**
	 * Enable menubar items according to loaded database.
	 *
	 * @param False if database is loaded
	 */
	private void setMenuBar(boolean open) {
   		openMenuItem.setEnabled(open);
	    closeMenuItem.setEnabled(!open);
		csvMenuItem.setEnabled(!open);
		categoriesMenuItem.setEnabled(!open);
		convertMenuItem.setEnabled(!open);
	}

	/**
	 * Enable buttons according to loaded database.
	 *
	 * @param enabled True if database is loaded
	 */
	private void enableButtons(boolean enabled) {
	     delEntry.setEnabled(false);
	     saveEntry.setEnabled(false);
	     newEntry.setEnabled(enabled);

	     enableFields(enabled);
	}

	/**
	 * Enable entry fields according to loaded database.
	 *
	 * @param flag True if database is loaded
	 */
	private void enableFields(boolean flag) {
		currentCategory.setEditable(false);
		currentTitle.setEditable(flag);
		currentAccountName.setEditable(flag);
		currentPassword.setEditable(flag);
		currentNotes.setEditable(flag);
	}

	/**
	 * Enable button lock according to loaded database and status of password timeout.
	 *
	 * @param locked True if application is locked
	 * @param enabled True if button "Lock" should be enabled
	 */
	private void setBtnLock(boolean locked, boolean enabled) {
		btnLock.setText(locked ? "Unlock" : "Lock");
		btnLock.setEnabled(enabled);
		this.locked = locked;

		saveEntry.setBackground(null);
	}

	// categories -----------------------------------------------------
	/**
	 * Setup category filter combobox.
	 *
	 * @param myCategories Categories loaded with Keyring database
	 */
	//private void setupCategories(Vector<String> myCategories) { // Java 1.5
	private void setupCategories(Vector myCategories) {
		// categoryList
		//Vector<String> displayCategories; // Java 1.5
		Vector displayCategories;
		if(myCategories != null)
			//displayCategories = new Vector<String>(myCategories); // Java 1.5
			displayCategories = (Vector)myCategories.clone();
		else
			//displayCategories = new Vector<String>(); // Java 1.5
			displayCategories = new Vector();

		displayCategories.add(0, "All");
		categoryList.setModel(new DefaultComboBoxModel(displayCategories));

		// currentCategory
		Vector currentCategories;
		if(myCategories != null)
			currentCategories = (Vector)myCategories.clone();
		else
			currentCategories = new Vector();

		currentCategory.setModel(new DefaultComboBoxModel(currentCategories));
	}

	// password -------------------------------------------------------
	/**
	 * Show password dialog and set Keyring database password.
	 *
	 * @return False if dialog cancelled (Boolean)
	 */
	private boolean checkPassword() {
		try {
			model.crypto.setPassword(getPasswordDialog());
			timeoutThread.restartTimeout();
			return true;
		}
		catch(Exception e) {
			msgError(e, "checkPassword", false);
			timeoutThread.setTimeout(); // timed out
			return false;
		}
	}

	/**
	 * Show password dialog.
	 *
	 * @return Password
	 */
	private char[] getPasswordDialog() throws Exception {
		PasswordDialog pwdDlg = new PasswordDialog(frame);
		pwdDlg.pack();
		pwdDlg.setVisible(true);

		return pwdDlg.getPassword();
	}

 	// showEntry ------------------------------------------------------
	/**
	 * Show entry and check password timeout.
	 */
	private void showEntry() {
		DefaultMutableTreeNode node = dynTree.getLastNode();

		if(locked == true) {
			return;
		}

		try {
			Date ende = timeoutThread.getEndDate();
			if(ende == null) { // timed out
				clearEntry();

				setBtnLock(true, true);
				enableButtons(false);
				saveEntry.setBackground(null);

				return;
			}
			else {
				timeoutThread.restartTimeout();
			}

			// no entry
			if(node == null || !(node.isLeaf()) || node.isRoot()) {
				clearEntry();

				saveEntry.setEnabled(false);
				saveEntry.setBackground(null);
				delEntry.setEnabled(false);

				return;
			}

			Object nodeInfo = node.getUserObject();

			Entry e = (Entry)nodeInfo;

			// set text fields according to entry
			// if categoryname was deleted, show first category "no category"
			if(currentCategory.getItemCount() > e.getCategory()) {
				currentCategory.setSelectedIndex(e.getCategory());
			}
			else {
				currentCategory.setSelectedIndex(0); // no category
			}

			currentTitle.setText(e.getTitle());
			currentAccountName.setText(e.getAccount());
			currentNotes.setText(e.getNotes());
			currentPassword.setText(e.getPassword());
			currentDate.setText(e.getDate());

			// initialize buttons
			textFieldChanged = false;
			saveEntry.setEnabled(false);
			saveEntry.setBackground(null);
			delEntry.setEnabled(true);
		}
		catch(Exception e) {
			msgError(e, "showEntry", true);

			try {
				loadDatabase(null);
			}
			catch(Exception ignore) {};
		}
	}

	/**
	 * Clear entry text fields.
	 */
	private void clearEntry() {
		//currentCategory.setSelectedIndex(0);
		currentTitle.setText("");
		currentAccountName.setText("");
		currentPassword.setText("");
		currentNotes.setText("");
		currentDate.setText("");
	}

	// ----------------------------------------------------------------
	// listener menubar -----------------------------------------------
	// ----------------------------------------------------------------

	/**
	 * MenuItem Open: shows a File dialog and load a Keyring database.
	 */
	public class OpenListener implements ActionListener {
		protected Editor editor;

		/**
		 * Default constructor.
		 *
		 * @param editor Reference to class Editor
		 */
		protected OpenListener(Editor editor) {
			this.editor = editor;
		}

		/**
		 * This method opens a file dialog and loads the choosen database.
		 *
		 * @param e the ActionEvent to process
		 */
		public void actionPerformed(ActionEvent e) {
			JFileChooser chooser = new JFileChooser();

			chooser.setDialogTitle("Open Keyring database");
			chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
			chooser.setMultiSelectionEnabled(false);
			chooser.setCurrentDirectory(previousDirectory);

			int returnVal = chooser.showOpenDialog(editor.frame);

			if(returnVal == JFileChooser.APPROVE_OPTION) {
				try {
					File selectedFile = chooser.getSelectedFile();
					previousDirectory = selectedFile.getParentFile();
					pdbFilename = selectedFile.getCanonicalPath();
					editor.loadDatabase(pdbFilename);
				}
				catch(Exception ex) {
					msgError(ex, "Open Keyring database", false);

					try {
						editor.loadDatabase(null);
					}
					catch(Exception ignore) {};
				}
			}
		}
	}

	/**
	 * MenuItem Close: close Keyring database.
	 */
	public class CloseListener implements ActionListener {
		protected Editor editor;

		/**
		 * Default constructor.
		 *
		 * @param editor Reference to class Editor
		 */
		protected CloseListener(Editor editor) {
			this.editor = editor;
		}

		/**
		 * This method closes the loaded database.
		 *
		 * @param e the ActionEvent to process
		 */
		public void actionPerformed(ActionEvent e) {
			try {
				editor.loadDatabase(null);
			}
			catch(Exception ignore) {};
		}
	}

	/**
	 * MenuItem Quit: exit Application.
	 */
	public class QuitListener implements ActionListener {
		protected Editor editor;

		/**
		 * Default constructor.
		 *
		 * @param editor Reference to class Editor
		 */
		protected QuitListener(Editor editor) {
			this.editor = editor;
		}

		/**
		 * This method ends the application.
		 *
		 * @param e the ActionEvent to process
		 */
		public void actionPerformed(ActionEvent e) {
			System.exit(0);
		}
	}

	/**
	 * MenuItem Edit categories: shows the categories dialog for editing category-names.
	 */
	public class editCategoriesListener implements ActionListener {
		protected Editor editor;

		/**
		 * Default constructor.
		 *
		 * @param editor Reference to class Editor
		 */
		protected editCategoriesListener(Editor editor) {
			this.editor = editor;
		}

		/**
		 * This method opens the categories dialog and updates the category combo boxes.
		 *
		 * @param e the ActionEvent to process
		 */
		public void actionPerformed(ActionEvent e) {
			if(locked == true) {
				msgInformation("Unlock application first.");
				return;
			}

			// show category dialog
			CategoriesDialog catDialog = new CategoriesDialog(frame, model.getCategories());

			catDialog.pack();
			catDialog.setVisible(true);

			// update category combo boxes
			//Vector<String> newCategories = catDialog.getNewCategories(); // Java 1.5
			Vector newCategories = catDialog.getNewCategories();
			if(newCategories != null) { // changed categories
				setupCategories(newCategories);
				model.setCategories(newCategories);

				// save database
				try {
					editor.model.saveData(pdbFilename);
				}
				catch(Exception ex) {
					msgError(ex, "Could not save entries to " + pdbFilename, false);
				}
			}
		}
	}

	/**
	 * MenuItem Save database to csv: save the loaded Keyring database as a CSV file.
	 */
	public class csvListener implements ActionListener {
		protected Editor editor;

		/**
		 * Default constructor.
		 *
		 * @param editor Reference to class Editor
		 */
		protected csvListener(Editor editor) {
			this.editor = editor;
		}

		/**
		 * This method opens a file dialog and saves the database entries to a csv file.
		 *
		 * @param e the ActionEvent to process
		 */
		public void actionPerformed(ActionEvent e) {
			if(locked == true) {
				msgInformation("Unlock application first.");
				return;
			}

			// show File dialog
			JFileChooser chooser = new JFileChooser();

			chooser.setDialogTitle("Save Keyring database to CSV-File");
			chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
			chooser.setMultiSelectionEnabled(false);
			chooser.setCurrentDirectory(previousDirectory);

			int returnVal = chooser.showSaveDialog(editor.frame);

			if(returnVal == JFileChooser.APPROVE_OPTION) {
				try {
					File selectedFile = chooser.getSelectedFile();
					previousDirectory = selectedFile.getParentFile();
					String csvFilename = selectedFile.getCanonicalPath();

					// check if file exists
					boolean ok = checkFile(csvFilename);

					if(ok == true) {
						// save entries to csv file
						editor.model.saveEntriesToFile(csvFilename);

						msgInformation("Entries saved to: " + model.getCsvFilename());
					}
				}
				catch(Exception ex) {
					msgError(ex, "Could not save entries to " + model.getCsvFilename(), false);
				}
			}
		}
	}

	/**
	 * MenuItem Convert database: shows convert dialog and convert loaded database.
	 */
	public class convertListener implements ActionListener {
		protected Editor editor;

		/**
		 * Default constructor.
		 *
		 * @param editor Reference to class Editor
		 */
		protected convertListener(Editor editor) {
			this.editor = editor;
		}

		/**
		 * This method opens the convert dialog and closes the loaded database.
		 *
		 * @param e the ActionEvent to process
		 */
		public void actionPerformed(ActionEvent e) {
			if(locked == true) {
				msgInformation("Unlock application first.");
				return;
			}

			// show convert dialog
			ConvertDialog catDialog = new ConvertDialog(editor.frame, editor.model, editor.model.pdbVersion);

			catDialog.pack();
			catDialog.setVisible(true);

			if(catDialog.getCancelled() == false) {
				msgInformation("Database converted.");

				// close old database
				try {
					editor.loadDatabase(null);
				}
				catch(Exception ignore) {};
			}
		}
	}

	/**
	 * MenuItem New minimal database: generates a new minimal database (database format 4)
	 */
	public class newListener implements ActionListener {
		protected Editor editor;

		/**
		 * Default constructor.
		 *
		 * @param editor Reference to class Editor
		 */
		protected newListener(Editor editor) {
			this.editor = editor;
		}

		/**
		 * This method opens a file dialog and generates a new minimal database.
		 *
		 * @param e the ActionEvent to process
		 */
		public void actionPerformed(ActionEvent e) {
			if(locked == true) {
				msgInformation("Unlock application first.");
				return;
			}

			// show File dialog
			JFileChooser chooser = new JFileChooser();

			chooser.setDialogTitle("Generate new minimal database");
			chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
			chooser.setMultiSelectionEnabled(false);
			chooser.setCurrentDirectory(previousDirectory);

			int returnVal = chooser.showSaveDialog(editor.frame);

			if(returnVal == JFileChooser.APPROVE_OPTION) {
				try {
					File selectedFile = chooser.getSelectedFile();
					previousDirectory = selectedFile.getParentFile();
					String csvFilename = selectedFile.getCanonicalPath();

					// check if file exists
					boolean ok = checkFile(csvFilename);

					if(ok == true) {
						// generate new minimal database
						model.writeNewDatabase(csvFilename);

						msgInformation("New minimal database with password 'test' generated.");
					}
				}
				catch(Exception ex) {
					msgError(ex, "Could not generate file " + model.getCsvFilename(), false);
				}
			}
		}
	}

	/**
	 * MenuItem About: show Copyright information.
	 */
	public class AboutListener implements ActionListener {
		protected Editor editor;

		/**
		 * Default constructor.
		 *
		 * @param editor Reference to class Editor
		 */
		protected AboutListener(Editor editor) {
			this.editor = editor;
		}

		/**
		 * This method shows a copyright information dialog.
		 *
		 * @param e the ActionEvent to process
		 */
		public void actionPerformed(ActionEvent e) {
			JOptionPane.showMessageDialog(editor.frame,
				gui.FRAMETITLE + " " + "v" + editor.VERSION +
				"\n\nCopyright 2005 Markus Griessnig\n" +
				"Vienna University of Technology\n" +
				"Institute of Computer Technology\n\n" +
				"KeyringEditor is based on:\n" +
				"Java Keyring v0.6\n" +
				"Copyright 2004 Frank Taylor <keyring@lieder.me.uk>\n\n" +
				"These programs are distributed in the hope that they will be useful,\n" +
				"but WITHOUT ANY WARRANTY; without even the implied warranty of\n" +
				"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n" +
				"See the GNU General Public License for more details.\n",
				"About",
				JOptionPane.INFORMATION_MESSAGE);
		}
	}

	// ----------------------------------------------------------------
	// listener buttons -----------------------------------------------
	// ----------------------------------------------------------------

	// new
	/**
	 * Button New: show edit dialog and save new entry to database.
	 */
	public class newEntryListener implements ActionListener {
		protected Editor editor;

		/**
		 * Default constructor.
		 *
		 * @param editor Reference to class Editor
		 */
		protected newEntryListener(Editor editor) {
			this.editor = editor;
		}

		/**
		 * This method opens the new entry dialog and and saves the new entry to database.
		 *
		 * @param e the ActionEvent to process
		 */
		public void actionPerformed(ActionEvent e) {
			// show edit dialog
			EditDialog editDlg = new EditDialog(editor.frame, editor.model.getCategories());
			editDlg.pack();
			editDlg.setVisible(true);

			Object[] buffer = editDlg.getNewEntry();
			byte[] record = null;
			byte[] ciphertext;
			int recordLength = 0;
			int newEntryId = editor.model.getEntriesSize() + 1;
			int ivLen = 8; // for TripleDES
			int id; // unique id

			try {
				// new entry
				if(buffer[0] != null) {
					// set record format & IV length
					switch(model.pdbVersion) {
						case 4:
							record = Model.toRecordFormat4( // account + password + notes
								(String)buffer[2] + "\0" +
								(String)buffer[3] + "\0" +
								(String)buffer[4] + "\0");
							break;
						case 5:
							record = Model.toRecordFormat5(
								(String)buffer[2],
								(String)buffer[3],
								(String)buffer[4]);

							if(model.crypto.type != 1) { // TripleDES
								ivLen = 16; // AES128, AES256
							}

							break;
					}

					// encrypt record
					ciphertext = editor.model.crypto.encrypt(record);

					int len = ((String)buffer[1]).length() + ciphertext.length - 16;
					switch(model.pdbVersion) {
						case 4: recordLength = len + 1;	break;
						case 5:	recordLength = len + 4 + (len % 2) + ivLen;	break;
					}

					// get new unique id
					id = editor.model.getNewUniqueId();

					// new entry object
					Entry myEntry = new Entry(
						newEntryId,
						(String)buffer[1],
						((Integer)buffer[0]).intValue(),
						Model.sliceBytes(ciphertext, 16, ciphertext.length - 16),
						editor.model.crypto,
						((Integer)buffer[0]).intValue() | 0x40,
						id,
						recordLength,
						Model.sliceBytes(ciphertext, 0, ivLen)); // Keyring database format 4: IV ignored

					// register new entry to vector entries
					editor.model.addEntry((Object)myEntry);

					// update tree view
					editor.dynTree.populate();

					// save database
					editor.model.saveData(pdbFilename);

					// mg
					// show new entry
					editor.dynTree.show((Object)myEntry);
				}
			}
			catch(Exception ex) {
				msgError(ex, "newEntryListener", true);
			}
		}
	}

	// save
	/**
	 * Button Save: save changed entry to database.
	 */
	public class saveEntryListener implements ActionListener {
		protected Editor editor;

		/**
		 * Default constructor.
		 *
		 * @param editor Reference to class Editor
		 */
		public saveEntryListener(Editor editor) {
			this.editor = editor;
		}

		/**
		 * This method saves a changed entry to database.
		 *
		 * @param e the ActionEvent to process
		 */
		public void actionPerformed(ActionEvent e) {
			DefaultMutableTreeNode node = editor.dynTree.getLastNode();
			byte[] record = null;
			byte[] ciphertext;
			int recordLength = 0;
			int ivLen = 8; // for TripleDES

			try {
				// last selected tree node
				if(node != null) {
					// get updated entry
					Entry myEntry = editor.dynTree.getEntry(node);

					char[] temp = editor.currentPassword.getPassword();

					// set record format & IV length
					switch(model.pdbVersion) {
						case 4:
							record = Model.toRecordFormat4(
								editor.currentAccountName.getText() + "\0" + // account + password + notes
								(new String(temp)) + "\0" +
								editor.currentNotes.getText() + "\0");
							break;
						case 5:
							record = Model.toRecordFormat5(
								editor.currentAccountName.getText(),
								(new String(temp)),
								editor.currentNotes.getText());

							if(model.crypto.type != 1) { // TripleDES
								ivLen = 16; // AES128, AES256
							}

							break;
					}

					// encrypt record
					ciphertext = editor.model.crypto.encrypt(record);

					int len = editor.currentTitle.getText().length() + ciphertext.length - 16;
					switch(model.pdbVersion) {
						case 4: recordLength = len + 1;	break;
						case 5:	recordLength = len + 4 + (len % 2) + ivLen;	break;
					}

					// save changes
					myEntry.title = editor.currentTitle.getText();
					myEntry.encrypted = Model.sliceBytes(ciphertext, 16, ciphertext.length - 16);
					myEntry.category = editor.currentCategory.getSelectedIndex();
					myEntry.attribute = editor.currentCategory.getSelectedIndex() | 0x40; // record ok
					myEntry.recordLength = recordLength;
					myEntry.iv = Model.sliceBytes(ciphertext, 0, ivLen);

					// update tree view
					editor.dynTree.populate();

					// save database
					editor.model.saveData(pdbFilename);

					msgInformation("Entries saved to: " + editor.pdbFilename);
				}
			}
			catch(Exception ex) {
				msgError(ex, "saveEntryListener", true);
			}
		}
	}

	// del
	/**
	 * Button Delete: delete current entry.
	 */
	public class delEntryListener implements ActionListener {
		protected Editor editor;

		/**
		 * Default constructor.
		 *
		 * @param editor Reference to class Editor
		 */
		public delEntryListener(Editor editor) {
			this.editor = editor;
		}

		/**
		 * This method removes a entry from database.
		 *
		 * @param e the ActionEvent to process
		 */
		public void actionPerformed(ActionEvent e) {
			DefaultMutableTreeNode node = editor.dynTree.getLastNode();

			if(node != null) {
				Entry myEntry = editor.dynTree.getEntry(node);

				// delete entry
				editor.model.removeEntry((Object)myEntry);

				// update tree view
				editor.dynTree.populate();

				// save changes
				try {
					editor.model.saveData(pdbFilename);

					msgInformation("Entry " + myEntry.getEntryId() + " deleted. Database " + editor.pdbFilename + " updated.");
				}
				catch(Exception ex) {
					msgError(ex, "delEntryListener", false);
				}
			}
		}
	}

	// ----------------------------------------------------------------
	// listener -------------------------------------------------------
	// ----------------------------------------------------------------

	// textfields
	/**
	 * DocumentListener: check if entry is updated and set button "Save" according.
	 */
	public class documentListener implements DocumentListener {
		protected Editor editor;

		protected documentListener(Editor editor) {
			this.editor = editor;
		}

		public void insertUpdate(DocumentEvent e) {
			updateLog(e, "insert");
		}

		public void removeUpdate(DocumentEvent e) {
			updateLog(e, "remove");
		}

		public void changedUpdate(DocumentEvent e) {
			updateLog(e, "changed");
		}

		public void updateLog(DocumentEvent e, String action) {
			if(editor.textFieldChanged == false && editor.locked == false) {
				editor.textFieldChanged = true;

				editor.saveEntry.setEnabled(true);
				editor.saveEntry.setBackground(Color.YELLOW);
			}
		}
	}

	// tree
	/**
	 * TreeSelectionListener: show selected entry.
	 */
	public class treeSelectionListener implements TreeSelectionListener {
		protected Editor editor;

		/**
		 * Default constructor.
		 *
		 * @param editor Reference to class Editor
		 */
		protected treeSelectionListener(Editor editor) {
			this.editor = editor;
		}

		/**
		 * This method shows the selected entry.
		 *
		 * @param e the ActionEvent to process
		 */
		public void valueChanged(TreeSelectionEvent e) {
			editor.showEntry();
		}
	}

	/**
	 * CategorySelectionListener: filter tree view according to selected category.
	 */
	public class CategorySelectionListener implements ActionListener {
		protected Editor editor;

		/**
		 * Default constructor.
		 *
		 * @param editor Reference to class Editor
		 */
		public CategorySelectionListener(Editor editor) {
			this.editor = editor;
		}

		/**
		 * This method filters the tree view according to selected category.
		 *
		 * @param e the ActionEvent to process
		 */
		public void actionPerformed(ActionEvent e) {

			editor.dynTree.setCategoryFilter(editor.categoryList.getSelectedIndex());

			editor.showEntry();
		}
	}

	/**
	 * currentCategorySelectionListener: check if entry category is changed an set button "Save" according.
	 */
	public class currentCategorySelectionListener implements ActionListener {
		protected Editor editor;

		/**
		 * Default constructor.
		 *
		 * @param editor Reference to class Editor
		 */
		public currentCategorySelectionListener(Editor editor) {
			this.editor = editor;
		}

		/**
		 * This method sets button "Save" according to state of variable locked and textFieldChanged.
		 *
		 * @param e the ActionEvent to process
		 */
		public void actionPerformed(ActionEvent e) {
			//System.out.println(editor.textFieldChanged);

			if(editor.textFieldChanged == false && editor.locked == false) {
				editor.textFieldChanged = true;

				editor.saveEntry.setEnabled(true);
				editor.saveEntry.setBackground(Color.YELLOW);
			}
		}
	}

	/**
	 * PasswordShowListener: show entry password in clear text according to check box "Hide passwords?"
	 */
	public class PasswordShowListener implements ActionListener {
		protected Editor editor;

		/**
		 * Default constructor.
		 *
		 * @param editor Reference to class Editor
		 */
		protected PasswordShowListener(Editor editor) {
			this.editor = editor;
		}

		/**
		 * This method shows password in clear text according to variable showPassword.
		 *
		 * @param e the ActionEvent to process
		 */
		public void actionPerformed(ActionEvent e) {
			if(showPassword == false) {
				showPassword = true;
				editor.currentPassword.setEchoChar('\0'); // show password in plaintext
			}
			else {
				showPassword = false;
				editor.currentPassword.setEchoChar('*');
			}
		}
	}

	/**
	 * Button Lock / Unlock: set buttons according to variable locked.
	 */
	public class PasswordLockListener implements ActionListener {
		protected Editor editor;

		/**
		 * Default constructor.
		 *
		 * @param editor Reference to class Editor
		 */
		protected PasswordLockListener(Editor editor) {
			this.editor = editor;
		}

		/**
		 * This method sets the button "Lock / Unlock" according to variable locked.
		 *
		 * @param e the ActionEvent to process
		 */
		public void actionPerformed(ActionEvent e) {
			if(editor.locked == false) {
				editor.setBtnLock(true, true);
				editor.enableButtons(false);
				editor.clearEntry();
			}
			else {
				if(editor.checkPassword() == true) {
					editor.setBtnLock(false, true);
					editor.enableButtons(true);
					editor.showEntry();
				}
			}
		}
	}
}
