/*
KeyringEditor

Copyright 2004 Markus Griessnig
Vienna University of Technology
Institute of Computer Technology

KeyringEditor is based on:
Java Keyring v0.6
Copyright 2004 Frank Taylor <keyring@lieder.me.uk>

These programs are distributed in the hope that they will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.
*/

// DynamicTree.java

// 22.11.2004

// 24.11.2004: added getTree()
// 08.12.2004: populateTree() updated

import java.awt.*;
import java.util.*;
import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;

/**
 * This class is used to view and manipulate the entries in the tree view.
 */
public class DynamicTree extends JPanel {
	// ----------------------------------------------------------------
	// variables
	// ----------------------------------------------------------------

	/**
	 * Root node
	 */
	private DefaultMutableTreeNode rootNode;

	private DefaultTreeModel treeModel;

	/**
	 * A control that displays a set of hierarchical data as an outline
	 */
	 private JTree tree;

	/**
	 * Reference to the class Editor
	 */
	private Editor editor;

	/**
	 * Category-filter (0 = show all)
	 */
	private int filterCategory = 0;

	// ----------------------------------------------------------------
	// constructor
	// ----------------------------------------------------------------
	/**
	 * Default constructor generates tree view.
	 *
	 * @param editor Reference to the class Editor
	 */
	 public DynamicTree(Editor editor) {
		super(new GridLayout(1,0));

		this.editor = editor;

		rootNode = new DefaultMutableTreeNode("");
		treeModel = new DefaultTreeModel(rootNode);

		tree = new JTree(treeModel);
		tree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
		tree.setRootVisible(true);

		JScrollPane scrollPane = new JScrollPane(tree);
		add(scrollPane);
	}

	// ----------------------------------------------------------------
	// public ---------------------------------------------------------
	// ----------------------------------------------------------------

	// get Entry
	/**
	 * This method selects a tree node according to the entry object.
	 *
	 * @param newentry Entry object
	 */
	public void show(Object newentry) {
		Object nodeInfo;
		
		DefaultMutableTreeNode node = rootNode;

		for(Enumeration enum1 = rootNode.depthFirstEnumeration(); enum1.hasMoreElements(); ) {
			node = (DefaultMutableTreeNode)enum1.nextElement();
			
			if(node != null) {
				if(node.isLeaf()) {
					nodeInfo = node.getUserObject();
					
					Entry e1 = (Entry)nodeInfo;
					Entry e2 = (Entry)newentry;
					
					if(e1.getEntryId() == e2.getEntryId()) {
						// New entry found
						tree.setSelectionPath(new TreePath(node.getPath()));
					}
				}
			}
		}
	}

	/**
	 * This method returns variable tree.
	 *
	 * @return variable tree
	 */
	public JTree getTree() {
		return tree;
	}

	// category filter
	/**
	 * This method sets the category-filter and refreshes the tree view.
	 *
	 * @return True if category changed otherwise false
	 */
	public boolean setCategoryFilter(int filterCategory) {
		boolean changed = (filterCategory != this.filterCategory);
		this.filterCategory = filterCategory;

		populate();

		return changed;
	}

	/**
	 * This method sets the category-filter to zero (show all) and refreshes the tree view.
	 */
	public void clearFilter() {
		filterCategory = 0;
		populate();
	}

	// get Entry
	/**
	 * This method returns the entry data of the tree node.
	 *
	 * @param node tree-node
	 *
	 * @return Entry data of tree-node
	 */
	public Entry getEntry(DefaultMutableTreeNode node) {
		Object nodeInfo = node.getUserObject();

		return (Entry)nodeInfo;
	}

	// get last selected path component
	/**
	 * This method returns the last selected tree node.
	 *
	 * @return DefaultMutableTreeNode or null if node is not a leaf
	 */
	public DefaultMutableTreeNode getLastNode() {
		DefaultMutableTreeNode temp = (DefaultMutableTreeNode)tree.getLastSelectedPathComponent();

		if(temp != null) {
			if(temp.isLeaf()) {
				return temp;
			}
		}

		return null;
	}

	// reload TreeModel
	/**
	 * This method reloads the tree model.
	 */
	public void reload() {
		treeModel.reload();
	}

	// remove all children
	/**
	 * This method removes all tree nodes and reloads the tree.
	 */
	public void clear() {
		rootNode.setUserObject("");
		rootNode.removeAllChildren();
		treeModel.reload();
	}

	// remove object
	/**
	 * This method removes the current selected node from the tree.
	 */
	public void removeCurrentNode() {
		TreePath currentSelection = tree.getSelectionPath();
		if(currentSelection != null) {
			DefaultMutableTreeNode currentNode = (DefaultMutableTreeNode)(currentSelection.getLastPathComponent());
			MutableTreeNode parent = (MutableTreeNode)(currentNode.getParent());
			if(parent != null) {
				treeModel.removeNodeFromParent(currentNode);
				return;
			}
		}
	}

	// populate tree
	/**
	 * This method populates the tree with entries.
	 */
	public void populate() {
		int i;
		int start;
		String parent;
		String child;
		DefaultMutableTreeNode node;
		DefaultMutableTreeNode startNode;

		clear();

		//String[] dbType = {"TripleDES", "TripleDES", "AES128", "AES256"};
		rootNode.setUserObject(editor.getFilename());

		// sort Entries
		Model myModel = editor.getModel();
		if(myModel == null) {
			return;
		}

		Vector myEntries = (Vector)myModel.getEntries().clone();
		Collections.sort(myEntries); // sort entries by title  // Java 1.5 - 2 warnings

		for(Enumeration e = myEntries.elements(); e.hasMoreElements(); ) {
			Dummy dummy;

			Entry entry = (Entry)e.nextElement();

			if(filterCategory != 0) {
				if((filterCategory - 1) != entry.getCategory()) {
					continue;
				}
			}

			start = 0;
			startNode = rootNode;

			// search for separator
			String title = entry.getTitle();
			while((i = title.indexOf(editor.getSeparator(), start)) != -1) {
				parent = title.substring(start, i);
				child = title.substring(i + 1, title.length());
				start = i + 1;

				node = searchForBranchNode(startNode, parent);

				if(node == null) {
					// new branch node
					dummy = new Dummy(parent);
					node = addObject(startNode, (Object)dummy, true); // parent node
				}

				startNode = node;
			}

			addObject(startNode, (Object)entry, true); // leaf
		}
	}

	// add object
	/**
	 * This method adds a node to the tree.
	 *
	 * @param parent Parent node
	 * @param child Object to add
	 * @param shouldBeVisible show new node
	 */
	public DefaultMutableTreeNode addObject(
		DefaultMutableTreeNode parent,
		Object child,
		boolean shouldBeVisible) {

		DefaultMutableTreeNode childNode = new DefaultMutableTreeNode(child);

		if(parent == null) {
			parent = rootNode;
		}

		treeModel.insertNodeInto(childNode, parent, parent.getChildCount());

		if(shouldBeVisible) {
			tree.scrollPathToVisible(new TreePath(childNode.getPath()));
		}

		return childNode;
	}

	// ----------------------------------------------------------------
	// private --------------------------------------------------------
	// ----------------------------------------------------------------

	/**
	 * This method searches if category exists as a branch node.
	 *
	 * @param root Root node
	 * @param category Node to search for
	 */
	private static DefaultMutableTreeNode searchForBranchNode(DefaultMutableTreeNode root, String category) {
		Enumeration e = root.children();
		DefaultMutableTreeNode node;
		Dummy branch;

		while(e.hasMoreElements()) {
			node = (DefaultMutableTreeNode)e.nextElement();
			try {
				branch = (Dummy)node.getUserObject();

				if(category.equals(branch.toString())) {
					return node;
				}
			}
			catch(Exception ex) {}; // ignore Entry objects
		}

		return null;
	}

	// used for branche nodes
	/**
	 * This class is used to present branch nodes in the tree view.
	 */
	private static class Dummy {
		String title;

		/**
		 * Default constructor.
		 *
		 * @param title Branch node title
		 */
		public Dummy(String title) {
			this.title = title;
		}

		/**
		 * This method returns the branch node title.
		 */
		public String toString() {
			return this.title;
		}
	}
}
