/*
KeyringEditor

Copyright 2004 Markus Griessnig
Vienna University of Technology
Institute of Computer Technology

KeyringEditor is based on:
Java Keyring v0.6
Copyright 2004 Frank Taylor <keyring@lieder.me.uk>

These programs are distributed in the hope that they will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.
*/

// ConvertDialog.java

// 07.12.2004
// 26.01.2005: check for empty password

import javax.swing.event.*;
import java.awt.event.*;
import javax.swing.*;
import java.beans.*; // property change stuff
import java.util.*;
import java.awt.*;
import java.io.*;

/**
 * This dialog allows the user to convert a database to different keyring database formats.
 */
public class ConvertDialog extends JDialog implements ActionListener, PropertyChangeListener {
	// ----------------------------------------------------------------
	// variables
	// ----------------------------------------------------------------
	/**
	 * Reference to the Gui frame
	 */
	private Frame frame;

	private JComboBox cboCipher;
	private JComboBox cboIter;
	private JPasswordField password;
	//private Vector<String> cipher; // Java 1.5
	//private Vector<Integer> iter; // Java 1.5
	private Vector cipher;
	private Vector iter;

	/**
	 * Reference to the class Model
	 */
	private Model model;

	/**
	 * Database format of the loaded database
	 */
	private int fromFormat;

	private JOptionPane optionPane;

	private String btnString1 = "OK";
	private String btnString2 = "Cancel";

	/**
	 * True if button "Cancel" pressed, otherwise false
	 */
	private boolean cancelled = false;

	// ----------------------------------------------------------------
	// constructor
	// ----------------------------------------------------------------
	/**
	 * Default constructor generates Dialog.
	 *
	 * @param frame Reference to the Gui frame
	 * @param model Reference to class Model
	 * @param fromFormat Database format of the loaded database
	 */
	public ConvertDialog(Frame frame, Model model, int fromFormat) {
		super(frame, "Convert database", true);

		this.frame = frame;
		this.model = model;
		this.fromFormat = fromFormat;

		// Cipher
		//cipher = new Vector<String>(); // Java 1.5
		cipher = new Vector();

		cipher.add("TripleDES | Database format 4");
		cipher.add("TripleDES | Database format 5");
		cipher.add("AES128 | Database format 5");
		cipher.add("AES256 | Database format 5");

		cboCipher = new JComboBox(cipher);
		cboCipher.setModel(new DefaultComboBoxModel(cipher));

		// iterations
		//iter = new Vector<Integer>(); // Java 1.5
		iter = new Vector();
		iter.add(new Integer(50));
		iter.add(new Integer(250));
		iter.add(new Integer(500));
		iter.add(new Integer(1000));

		cboIter = new JComboBox(iter);
		cboIter.setModel(new DefaultComboBoxModel(iter));

		password = new JPasswordField(20);

		// labels
		String msgString1 = "Cipher: ";
		String msgString2 = "Iterations: ";
		String msgString3 = "New password: ";

		Object[] array = {msgString1, cboCipher, msgString2, cboIter, msgString3, password};

		Object[] options = {btnString1, btnString2};

		// generate dialog
		optionPane = new JOptionPane(array,
			JOptionPane.PLAIN_MESSAGE,
			JOptionPane.YES_NO_OPTION,
			null,
			options,
			options[0]);

		setContentPane(optionPane);

		setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent we) {
			/*
			 * Instead of directly closing the window,
			 * we're going to change the JOptionPane's
			 * value property.
			 */
			 optionPane.setValue(new Integer(JOptionPane.CLOSED_OPTION));
			}
		});

		addComponentListener(new ComponentAdapter() {
			public void componentShown(ComponentEvent ce) {
				password.requestFocusInWindow();
			}
		});

		//Register an event handler that puts the text into the option pane.
		//allCategories.addActionListener(this);

		//Register an event handler that reacts to option pane state changes.
		optionPane.addPropertyChangeListener(this);
	}

	// ----------------------------------------------------------------
	// public ---------------------------------------------------------
	// ----------------------------------------------------------------

	/**
	 * This method returns status of variable cancelled.
	 *
	 * @return variable cancelled
	 */
	public boolean getCancelled() {
		return cancelled;
	}

	/**
	 * This method is empty.
	 *
	 * @param e the ActionEvent to process
	 */
	public void actionPerformed(ActionEvent e) {
	}

	/**
	 * This method processes the pressed button.
	 * If button is OK database is converted to selected keyring database format and
	 * encrypted with new password and selected cryptographic function.
	 * Otherwise variable cancelled is set to true.
	 *
	 * @param e the PropertyChangeEvent to process
	 */
	public void propertyChange(PropertyChangeEvent e) {
		String prop = e.getPropertyName();

		if (isVisible()
			&& (e.getSource() == optionPane)
			&& (JOptionPane.VALUE_PROPERTY.equals(prop) ||
				JOptionPane.INPUT_VALUE_PROPERTY.equals(prop))) {

			Object value = optionPane.getValue();

			if(value == JOptionPane.UNINITIALIZED_VALUE) {
				//ignore reset
				return;
			}

			//Reset the JOptionPane's value.
			//If you don't do this, then if the user
			//presses the same button next time, no
			//property change event will be fired.
			optionPane.setValue(JOptionPane.UNINITIALIZED_VALUE);

			if(btnString1.equals(value)) {
				// check for empty password
				if(password.getPassword().length == 0) {
					JOptionPane.showMessageDialog(ConvertDialog.this, "Sorry, password must not be empty!\n",
                    	"Error", JOptionPane.ERROR_MESSAGE);
					cancelled = true;
                    password.requestFocusInWindow();
                    return;
                }

				// ok
				JFileChooser chooser = new JFileChooser();

				chooser.setDialogTitle("Convert Keyring database");
				chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
				chooser.setMultiSelectionEnabled(false);

				int returnVal = chooser.showSaveDialog(this.frame);

				if(returnVal == JFileChooser.APPROVE_OPTION) {
				try {
					File selectedFile = chooser.getSelectedFile();
					String toFilename = selectedFile.getCanonicalPath();

					int toFormat = 0;
					int type = cboCipher.getSelectedIndex();
					// 0 = TripleDES Format 4
					// 1 = TripleDES Format 5
					// 2 = AES-128 Format 5
					// 3 = AES-256 Format 5

					switch(type) {
						case 0: toFormat = 4; break;
						case 1:
						case 2:
						case 3: toFormat = 5; break;
					}

					// convert database
					model.convertDatabase(
						fromFormat,
						toFormat,
						toFilename,
						password.getPassword(),
						type,
						((Integer)cboIter.getSelectedItem()).intValue());
				}
				catch(Exception ex) {
					// cancel
					cancelled = true;

					JOptionPane.showMessageDialog(frame,
						"Convert database: " + ex.getMessage(),
						"Error",
						JOptionPane.ERROR_MESSAGE);

					ex.printStackTrace(System.err);
				}
				}
				else {
					// cancel
					cancelled = true;
				}
			}
			else {
				// cancel
				cancelled = true;
			}

			clearAndHide();
		}
	}

	// ----------------------------------------------------------------
	// private --------------------------------------------------------
	// ----------------------------------------------------------------

	/**
	 * This method hides the dialog.
	 */
	private void clearAndHide() {
		password.setText(null);
		setVisible(false);
	}
}
